<?php
/**
 * Scheduler Class
 *
 * @package OOPVulns
 */

namespace OOPVulns;

defined( 'ABSPATH' ) || exit;

/**
 * Handles scheduled scans and notifications.
 */
class Scheduler {
	/**
	 * Scanner instance.
	 *
	 * @var Scanner
	 */
	private $scanner;

	/**
	 * Notifications instance.
	 *
	 * @var Notifications
	 */
	private $notifications;

	/**
	 * Constructor.
	 *
	 * @param Scanner       $scanner Scanner instance.
	 * @param Notifications $notifications Notifications instance.
	 */
	public function __construct( Scanner $scanner, Notifications $notifications ) {
		$this->scanner       = $scanner;
		$this->notifications = $notifications;
	}

	/**
	 * Initialize scheduler hooks.
	 */
	public function init() {
		// Add custom schedules.
		add_filter( 'cron_schedules', array( $this, 'add_custom_schedules' ) );

		// Hook scan and notification actions.
		add_action( 'oopvulns_scan', array( $this, 'run_scheduled_scan' ) );
		add_action( 'oopvulns_notify', array( $this, 'run_scheduled_notification' ) );

		// Update schedules when settings change.
		add_action( 'oopvulns_schedules_update', array( $this, 'update_all_schedules' ) );

		// Ensure schedules are set up.
		$this->ensure_schedules();
	}

	/**
	 * Add custom cron schedules.
	 *
	 * @param array $schedules Existing schedules.
	 * @return array Modified schedules.
	 */
	public function add_custom_schedules( $schedules ) {
		$schedules['weekly'] = array(
			'interval' => WEEK_IN_SECONDS,
			'display'  => __( 'Once Weekly', 'oopvulns-vulnerability-scanner' ),
		);

		return $schedules;
	}

	/**
	 * Run scheduled vulnerability scan.
	 */
	public function run_scheduled_scan() {
		$results = $this->scanner->scan_all( false ); // Force fresh scan.
		$this->scanner->update_last_scan_time();
	}

	/**
	 * Run scheduled notification.
	 */
	public function run_scheduled_notification() {
		// Get cached scan results.
		$core    = get_option( 'oopvulns_core_result', array() );
		$plugins = get_option( 'oopvulns_plugins_result', array() );
		$themes  = get_option( 'oopvulns_themes_result', array() );

		if ( empty( $core ) && empty( $plugins ) && empty( $themes ) ) {
			// No scan results, run a scan first.
			$this->run_scheduled_scan();
			$core    = get_option( 'oopvulns_core_result', array() );
			$plugins = get_option( 'oopvulns_plugins_result', array() );
			$themes  = get_option( 'oopvulns_themes_result', array() );
		}

		$results = array(
			'core'    => $core,
			'plugins' => $plugins,
			'themes'  => $themes,
			'summary' => $this->scanner->get_scan_summary(),
		);

		// Send notification.
		$this->notifications->send_notification( $results, false );
	}

	/**
	 * Update all schedules (both scan and notification).
	 */
	public function update_all_schedules() {
		$this->update_scan_schedule();
		$this->update_notification_schedule();
	}

	/**
	 * Ensure schedules are properly set up.
	 */
	private function ensure_schedules() {
		// Schedule scan only if missing.
		if ( ! wp_next_scheduled( 'oopvulns_scan' ) ) {
			$this->update_scan_schedule();
		}

		// Schedule notifications only if missing.
		if ( ! wp_next_scheduled( 'oopvulns_notify' ) ) {
			$this->update_notification_schedule();
		}
	}

	/**
	 * Update scan schedule based on settings.
	 */
	public function update_scan_schedule() {
		$settings = get_option( 'oopvulns_settings', array() );
		
		// Clear existing schedule.
		wp_clear_scheduled_hook( 'oopvulns_scan' );

		// Get scan frequency from settings.
		$frequency = $settings['scan_frequency'] ?? 'daily';
		
		// Ensure valid frequency.
		if ( ! in_array( $frequency, array( 'daily', 'weekly' ), true ) ) {
			$frequency = 'daily';
		}

		// Schedule the scan at the next interval to avoid immediate runs on save.
		$schedules = wp_get_schedules();
		$interval  = $schedules[ $frequency ]['interval'] ?? DAY_IN_SECONDS;
		$next_run  = time() + absint( $interval );

		wp_schedule_event( $next_run, $frequency, 'oopvulns_scan' );
	}

	/**
	 * Update notification schedule based on settings.
	 */
	public function update_notification_schedule() {
		$settings = get_option( 'oopvulns_settings', array() );
		
		// Clear existing schedule.
		wp_clear_scheduled_hook( 'oopvulns_notify' );

		// Don't schedule if notifications are disabled.
		if ( empty( $settings['enabled'] ) ) {
			return;
		}

		$schedule = $settings['notification_schedule'] ?? 'weekly';
		$hour     = absint( $settings['notification_hour'] ?? 9 );
		$minute   = absint( $settings['notification_minute'] ?? 0 );

		// Calculate next run time.
		$next_run = $this->calculate_next_run( $schedule, $hour, $minute, $settings );

		if ( $next_run ) {
			wp_schedule_event( $next_run, $schedule, 'oopvulns_notify' );
		}
	}

	/**
	 * Calculate next run timestamp.
	 *
	 * @param string $schedule Schedule frequency.
	 * @param int    $hour Hour of day.
	 * @param int    $minute Minute of hour.
	 * @param array  $settings Plugin settings.
	 * @return int|false Timestamp or false.
	 */
	private function calculate_next_run( $schedule, $hour, $minute, $settings ) {
		$timezone = wp_timezone();
		$now      = new \DateTime( 'now', $timezone );
		$next     = new \DateTime( 'now', $timezone );
		$next->setTime( $hour, $minute, 0 );

		// If time has passed today, move to next occurrence.
		if ( $next <= $now ) {
			if ( $schedule === 'daily' ) {
				$next->modify( '+1 day' );
			} elseif ( $schedule === 'weekly' ) {
				$day = $settings['notification_day'] ?? 'monday';
				$next->modify( 'next ' . $day );
			}
		} elseif ( $schedule === 'weekly' ) {
			// Set to specific day of week.
			$day        = $settings['notification_day'] ?? 'monday';
			$target_day = gmdate( 'w', strtotime( $day ) );
			$current_day = $next->format( 'w' );
			
			if ( $current_day !== $target_day ) {
				$next->modify( 'next ' . $day );
			}
		}

		return $next->getTimestamp();
	}

	/**
	 * Clear all schedules.
	 */
	public function clear_schedules() {
		wp_clear_scheduled_hook( 'oopvulns_scan' );
		wp_clear_scheduled_hook( 'oopvulns_notify' );
	}
}
