<?php
/**
 * Scanner Class
 *
 * @package OOPVulns
 */

namespace OOPVulns;

defined( 'ABSPATH' ) || exit;

/**
 * Scans WordPress core, plugins, and themes for vulnerabilities.
 */
class Scanner {
	/**
	 * API client instance.
	 *
	 * @var API_Client
	 */
	public $api_client;

	/**
	 * Constructor.
	 *
	 * @param API_Client $api_client API client instance.
	 */
	public function __construct( API_Client $api_client ) {
		$this->api_client = $api_client;
	}

	/**
	 * Run a full vulnerability scan.
	 *
	 * @param bool $use_cache Whether to use cache.
	 * @return array Scan results.
	 */
	public function scan_all( $use_cache = true ) {
		return array(
			'core'    => $this->scan_core( $use_cache ),
			'plugins' => $this->scan_plugins( $use_cache ),
			'themes'  => $this->scan_themes( $use_cache ),
			'summary' => $this->get_scan_summary(),
		);
	}

	/**
	 * Scan WordPress core for vulnerabilities.
	 *
	 * @param bool $use_cache Whether to use cache.
	 * @return array Core scan results.
	 */
	public function scan_core( $use_cache = true ) {
		$version = get_bloginfo( 'version' );
		$vulns   = $this->api_client->get_core_vulnerabilities( $version, $use_cache );

		$result = array(
			'version'         => $version,
			'vulnerabilities' => $vulns ?: array(),
			'vulnerable'      => ! empty( $vulns ),
			'count'           => $vulns ? count( $vulns ) : 0,
		);

		// Cache the result.
		update_option( 'oopvulns_core_result', $result );

		return $result;
	}

	/**
	 * Scan all active plugins for vulnerabilities.
	 *
	 * @param bool $use_cache Whether to use cache.
	 * @return array Plugins scan results.
	 */
	public function scan_plugins( $use_cache = true ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins    = get_plugins();
		$results        = array();
		$total_vulns    = 0;

		foreach ( $all_plugins as $plugin_file => $plugin_data ) {
			$slug        = $this->get_plugin_slug( $plugin_file );
			$version     = $plugin_data['Version'];
			$vulns       = $this->api_client->get_plugin_vulnerabilities( $slug, $version, $use_cache );
			
			$results[ $slug ] = array(
				'name'            => $plugin_data['Name'],
				'version'         => $version,
				'slug'            => $slug,
				'file'            => $plugin_file,
				'active'          => is_plugin_active( $plugin_file ),
				'vulnerabilities' => $vulns ?: array(),
				'vulnerable'      => ! empty( $vulns ),
				'count'           => $vulns ? count( $vulns ) : 0,
			);

			if ( ! empty( $vulns ) ) {
				$total_vulns += count( $vulns );
			}
		}

		$summary = array(
			'total'             => count( $results ),
			'vulnerable'        => count( array_filter( $results, function( $r ) {
				return $r['vulnerable'];
			} ) ),
			'total_vulnerabilities' => $total_vulns,
			'plugins'           => $results,
		);

		// Cache the result.
		update_option( 'oopvulns_plugins_result', $summary );

		return $summary;
	}

	/**
	 * Scan all themes for vulnerabilities.
	 *
	 * @param bool $use_cache Whether to use cache.
	 * @return array Themes scan results.
	 */
	public function scan_themes( $use_cache = true ) {
		$all_themes     = wp_get_themes();
		$current_theme  = wp_get_theme();
		$results        = array();
		$total_vulns    = 0;

		foreach ( $all_themes as $slug => $theme ) {
			$version = $theme->get( 'Version' );
			$vulns   = $this->api_client->get_theme_vulnerabilities( $slug, $version, $use_cache );

			$results[ $slug ] = array(
				'name'            => $theme->get( 'Name' ),
				'version'         => $version,
				'slug'            => $slug,
				'active'          => $slug === $current_theme->get_stylesheet(),
				'vulnerabilities' => $vulns ?: array(),
				'vulnerable'      => ! empty( $vulns ),
				'count'           => $vulns ? count( $vulns ) : 0,
			);

			if ( ! empty( $vulns ) ) {
				$total_vulns += count( $vulns );
			}
		}

		$summary = array(
			'total'                 => count( $results ),
			'vulnerable'            => count( array_filter( $results, function( $r ) {
				return $r['vulnerable'];
			} ) ),
			'total_vulnerabilities' => $total_vulns,
			'themes'                => $results,
		);

		// Cache the result.
		update_option( 'oopvulns_themes_result', $summary );

		return $summary;
	}

	/**
	 * Get scan summary from cached results.
	 *
	 * @return array Scan summary.
	 */
	public function get_scan_summary() {
		$core    = get_option( 'oopvulns_core_result', array() );
		$plugins = get_option( 'oopvulns_plugins_result', array() );
		$themes  = get_option( 'oopvulns_themes_result', array() );

		$total_vulns = 0;
		$total_vulns += ! empty( $core['count'] ) ? $core['count'] : 0;
		$total_vulns += ! empty( $plugins['total_vulnerabilities'] ) ? $plugins['total_vulnerabilities'] : 0;
		$total_vulns += ! empty( $themes['total_vulnerabilities'] ) ? $themes['total_vulnerabilities'] : 0;

		return array(
			'total_vulnerabilities' => $total_vulns,
			'core_vulnerable'       => ! empty( $core['vulnerable'] ),
			'vulnerable_plugins'    => ! empty( $plugins['vulnerable'] ) ? $plugins['vulnerable'] : 0,
			'vulnerable_themes'     => ! empty( $themes['vulnerable'] ) ? $themes['vulnerable'] : 0,
			'last_scan'             => get_option( 'oopvulns_last_scan' ),
		);
	}

	/**
	 * Get plugin slug from plugin file path.
	 *
	 * @param string $plugin_file Plugin file path.
	 * @return string Plugin slug.
	 */
	public function get_plugin_slug( $plugin_file ) {
		$slug = dirname( $plugin_file );
		if ( '.' === $slug ) {
			$slug = basename( $plugin_file, '.php' );
		}
		return $slug;
	}

	/**
	 * Update last scan timestamp.
	 */
	public function update_last_scan_time() {
		update_option( 'oopvulns_last_scan', current_time( 'timestamp' ) );
	}

	/**
	 * Clear all scan results.
	 */
	public function clear_results() {
		delete_option( 'oopvulns_core_result' );
		delete_option( 'oopvulns_plugins_result' );
		delete_option( 'oopvulns_themes_result' );
		delete_option( 'oopvulns_last_scan' );
	}
}
