<?php
/**
 * Main Plugin Class
 *
 * @package OOPVulns
 */

namespace OOPVulns;

defined( 'ABSPATH' ) || exit;

/**
 * Main plugin orchestrator class.
 */
class Plugin {
	/**
	 * Singleton instance.
	 *
	 * @var Plugin
	 */
	private static $instance = null;

	/**
	 * API client instance.
	 *
	 * @var API_Client
	 */
	private $api_client;

	/**
	 * Scanner instance.
	 *
	 * @var Scanner
	 */
	private $scanner;

	/**
	 * Admin instance.
	 *
	 * @var Admin
	 */
	private $admin;

	/**
	 * Notifications instance.
	 *
	 * @var Notifications
	 */
	private $notifications;

	/**
	 * Get singleton instance.
	 *
	 * @return Plugin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Private constructor to prevent direct instantiation.
	 */
	private function __construct() {
		$this->api_client    = new API_Client();
		$this->scanner       = new Scanner( $this->api_client );
		$this->notifications = new Notifications();
		$this->admin         = new Admin( $this->scanner, $this->notifications );
	}

	/**
	 * Initialize the plugin.
	 */
	public function init() {
		// Initialize scheduler.
		$scheduler = new Scheduler( $this->scanner, $this->notifications );
		$scheduler->init();

		// Add settings link to plugins page.
		add_filter( 'plugin_action_links_' . OOPVULNS_BASENAME, array( $this, 'add_settings_link' ) );
	}

	/**
	 * Add settings link to plugin actions.
	 *
	 * @param array $links Existing links.
	 * @return array Modified links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = sprintf(
			'<a href="%s" aria-label="%s">%s</a>',
			esc_url( admin_url( 'tools.php?page=oopvulns-vulnerability-scanner' ) ),
			esc_attr__( 'Configure  OOPVulns  settings', 'oopvulns-vulnerability-scanner' ),
			esc_html__( 'Settings', 'oopvulns-vulnerability-scanner' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Plugin activation.
	 */
	public static function activate() {
		// Set default options.
		$default_options = array(
			'cache_hours'           => 12,
			'scan_frequency'        => 'daily',
			'notification_email'    => get_option( 'admin_email' ),
			'notification_schedule' => 'weekly',
			'notification_day'      => 'monday',
			'notification_hour'     => 9,
			'notification_minute'   => 0,
			'enabled'               => true,
		);

		if ( ! get_option( 'oopvulns_settings' ) ) {
			add_option( 'oopvulns_settings', $default_options );
		}

		// Clear any existing cache.
		self::clear_all_cache();
	}

	/**
	 * Plugin deactivation.
	 */
	public static function deactivate() {
		// Clear scheduled events.
		wp_clear_scheduled_hook( 'oopvulns_scan' );
		wp_clear_scheduled_hook( 'oopvulns_notify' );
		
		// Clear cache.
		self::clear_all_cache();
	}

	/**
	 * Clear all plugin cache.
	 */
	private static function clear_all_cache() {
		global $wpdb;
		
		// Delete all transients.
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
				$wpdb->esc_like( '_transient_oopvulns_' ) . '%'
			)
		);
		
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
				$wpdb->esc_like( '_transient_timeout_oopvulns_' ) . '%'
			)
		);
	}
}
