<?php
/**
 * Notifications Class
 *
 * @package OOPVulns
 */

namespace OOPVulns;

defined( 'ABSPATH' ) || exit;

/**
 * Handles email notifications for vulnerability alerts.
 */
class Notifications {
	/**
	 * Deduplicate vulnerabilities by product name and version.
	 *
	 * @param array $vulnerabilities List of vulnerabilities.
	 * @return array Array with 'items' and 'duplicates' keys.
	 */
	private function deduplicate_vulnerabilities( $vulnerabilities ) {
		if ( empty( $vulnerabilities ) || ! is_array( $vulnerabilities ) ) {
			return array(
				'items'      => array(),
				'duplicates' => array(),
				'count'      => 0,
			);
		}

		$items            = array();
		$duplicates       = array();
		$seen             = array();

		foreach ( $vulnerabilities as $vuln ) {
			$name = strtolower( trim( (string) ( $vuln['name'] ?? '' ) ) );

			// Remove bracketed content [like-this]
			$clean_name = preg_replace( '/\s*\[[^\]]*\]/', '', $name );

			// Extract version constraint (e.g., "<=", ">=", "=", etc. followed by version)
			$version_constraint = '';
			if ( preg_match( '/(<=|>=|<|>|=)\s*(\d+[\.\d]*)/', $clean_name, $matches ) ) {
				$version_constraint = $matches[1] . ' ' . $matches[2];
			}

			// Remove everything after typical description separators (-, (,  etc)
			// Keep only the product name and version constraint
			$product_part = preg_replace( '/\s*[-–]\s*.*/i', '', $clean_name );
			$product_part = preg_replace( '/\s*\(.*/i', '', $product_part );
			
			// Extract just the product name before version constraint
			$product_part = preg_replace( '/(<=|>=|<|>|=).*/i', '', $product_part );
			$product_name = strtolower( trim( $product_part ) );

			// Create dedup key: product name + version constraint
			$key = $product_name . '|' . $version_constraint;

			if ( isset( $seen[ $key ] ) ) {
				$duplicates[] = $vuln;
				continue;
			}

			$seen[ $key ] = true;
			$items[]      = $vuln;
		}

		return array(
			'items'      => $items,
			'duplicates' => $duplicates,
			'count'      => count( $duplicates ),
		);
	}

	/**
	 * Send notification email with scan results.
	 *
	 * @param array $scan_results Scan results.
	 * @param bool  $force Force send even if no vulnerabilities.
	 * @return bool True if sent successfully.
	 */
	public function send_notification( $scan_results, $force = false ) {
		$summary = $scan_results['summary'];

		// Don't send if no vulnerabilities (unless forced).
		if ( ! $force && $summary['total_vulnerabilities'] === 0 ) {
			return false;
		}

		$settings = get_option( 'oopvulns_settings', array() );
		$to       = $settings['notification_email'] ?? get_option( 'admin_email' );

		if ( empty( $to ) || ! is_email( $to ) ) {
			return false;
		}

		$subject = $this->get_subject( $summary );
		$message = $this->get_message( $scan_results );
		$headers = $this->get_headers();

		$sent = wp_mail( $to, $subject, $message, $headers );

		if ( $sent ) {
			update_option( 'oopvulns_last_notification', current_time( 'timestamp' ) );
		}

		return $sent;
	}

	/**
	 * Get email subject.
	 *
	 * @param array $summary Scan summary.
	 * @return string Email subject.
	 */
	private function get_subject( $summary ) {
		$site_name = get_bloginfo( 'name' );
		$count     = $summary['total_vulnerabilities'];

		if ( $count === 0 ) {
			return sprintf(
				/* translators: %s: Site name */
				__( '[%s] No Security Vulnerabilities Found', 'oopvulns-vulnerability-scanner' ),
				$site_name
			);
		}

		return sprintf(
			/* translators: 1: Site name, 2: Number of vulnerabilities */
			_n(
				'[%1$s] %2$d Security Vulnerability Detected',
				'[%1$s] %2$d Security Vulnerabilities Detected',
				$count,
				'oopvulns-vulnerability-scanner'
			),
			$site_name,
			$count
		);
	}

	/**
	 * Get email message body.
	 *
	 * @param array $scan_results Full scan results.
	 * @return string Email message.
	 */
	private function get_message( $scan_results ) {
		$site_name = get_bloginfo( 'name' );
		$site_url  = home_url();
		$admin_url = admin_url( 'tools.php?page=oopvulns-vulnerability-scanner' );
		$summary   = $scan_results['summary'];

		ob_start();
		?>
<!DOCTYPE html>
<html lang="<?php echo esc_attr( get_locale() ); ?>">
<head>
	<meta charset="UTF-8">
	<meta name="viewport" content="width=device-width, initial-scale=1.0">
	<style type="text/css">
		body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; line-height: 1.6; color: #2c2c2c; }
		table { border-collapse: collapse; width: 100%; }
		a { color: #000000; text-decoration: none; }
		.wrapper { background: #ffffff; max-width: 600px; margin: 0 auto; }
		.header { background: #ffffff; border-bottom: 2px solid #FCDD56; padding: 40px 30px; text-align: center; }
		.header h1 { margin: 0; font-size: 28px; font-weight: 700; color: #000000; }
		.header p { margin: 8px 0 0 0; font-size: 14px; color: #555555; font-weight: 500; }
		.content { padding: 40px 30px; background: #ffffff; }
		.alert { border-left: 4px solid #000000; padding: 16px; margin: 0 0 30px 0; border-radius: 0; background: #ffffff; }
		.alert-title { margin: 0; font-size: 16px; font-weight: 600; color: #000000; }
		.alert p { margin: 0; font-size: 14px; color: #555555; }
		.section { margin-bottom: 30px; }
		.section-title { font-size: 16px; font-weight: 700; color: #000000; margin: 0 0 16px 0; padding-bottom: 12px; border-bottom: 1px solid #e0e0e0; }
		.item-card { background: #ffffff; border-left: 3px solid #FCDD56; padding: 14px; margin-bottom: 12px; border-radius: 0; }
		.item-name { font-size: 14px; font-weight: 600; color: #000000; margin: 0 0 6px 0; }
		.item-meta { font-size: 13px; color: #666666; margin: 0; }
		.vulnerability { background: #ffffff; border-left: 3px solid #666666; padding: 10px; margin: 8px 0; border-radius: 0; font-size: 13px; }
		.vulnerability-name { font-weight: 600; color: #000000; display: block; margin-bottom: 4px; }
		.severity-badge { display: inline-block; background: #000000; color: #FCDD56; padding: 3px 8px; border-radius: 0; font-size: 11px; font-weight: 600; margin-right: 6px; }
		.duplicate-note { background: #ffffff; border-left: 3px solid #666666; padding: 10px; margin-top: 10px; border-radius: 0; font-size: 12px; color: #555555; }
		.cta-button { display: inline-block; background: #000000; color: #FCDD56; padding: 12px 32px; text-decoration: none; border-radius: 0; font-weight: 600; margin: 30px 0; }
		.cta-button:hover { background: #1a1a1a; }
		.footer { background: #ffffff; padding: 30px; text-align: center; border-top: 1px solid #e0e0e0; font-size: 12px; color: #666666; }
		.footer a { color: #000000; }
		.status-icon { font-size: 48px; line-height: 1; }
	</style>
</head>
<body style="margin: 0; padding: 20px; background: #f5f5f5;">
	<div class="wrapper">
		<!-- Header -->
		<div class="header">
			<div class="status-icon">
				<?php echo $summary['total_vulnerabilities'] > 0 ? '⚠️' : '✅'; ?>
			</div>
			<h1><?php echo $summary['total_vulnerabilities'] > 0 ? esc_html_e( 'Security Alert', 'oopvulns-vulnerability-scanner' ) : esc_html_e( 'Security Check Passed', 'oopvulns-vulnerability-scanner' ); ?></h1>
			<p><?php echo esc_html( $site_name ); ?></p>
		</div>

		<!-- Content -->
		<div class="content">
			<?php if ( $summary['total_vulnerabilities'] > 0 ) : ?>
				<!-- Alert Box -->
				<div class="alert">
					<p class="alert-title">
						<?php
						printf(
							/* translators: %d: Number of vulnerabilities */
							esc_html( _n(
								'%d security vulnerability detected',
								'%d security vulnerabilities detected',
								$summary['total_vulnerabilities'],
								'oopvulns-vulnerability-scanner'
							) ),
							absint( $summary['total_vulnerabilities'] )
						);
						?>
					</p>
					<p><?php esc_html_e( 'Review the details below and take action to secure your site.', 'oopvulns-vulnerability-scanner' ); ?></p>
				</div>

				<!-- WordPress Core Section -->
				<?php if ( ! empty( $scan_results['core']['vulnerable'] ) ) : ?>
					<?php
					$core_deduped = $this->deduplicate_vulnerabilities( $scan_results['core']['vulnerabilities'] ?? array() );
					$total_count  = count( $core_deduped['items'] ) + $core_deduped['count'];
					?>
					<div class="section">
						<h2 class="section-title">⚙️ WordPress Core</h2>
						<div class="item-card">
							<p class="item-name"><?php echo esc_html( get_bloginfo( 'version' ) ); ?></p>
							<p class="item-meta"><?php
								/* translators: %d: Number of vulnerabilities */
								printf( esc_html( _n( '%d vulnerability', '%d vulnerabilities', $total_count, 'oopvulns-vulnerability-scanner' ) ), absint( $total_count ) );
							?></p>
						</div>
						<?php foreach ( $core_deduped['items'] as $vuln ) : ?>
							<div class="vulnerability">
								<span class="vulnerability-name"><?php echo esc_html( $vuln['name'] ); ?></span>
								<span class="severity-badge"><?php echo esc_html( strtoupper( $vuln['severity'] ) ); ?></span>
								<span style="font-size: 12px; color: #666666;"><?php
									/* translators: %s: CVSS score number */
									printf( esc_html__( 'CVSS: %s', 'oopvulns-vulnerability-scanner' ), number_format( $vuln['cvss_score'], 1 ) );
								?></span>
							</div>
						<?php endforeach; ?>
						<?php if ( $core_deduped['count'] > 0 ) : ?>
							<div class="duplicate-note">
								+ <?php
								/* translators: %d: Number of duplicate entries */
								printf( esc_html( _n( '%d duplicate entry', '%d duplicate entries', $core_deduped['count'], 'oopvulns-vulnerability-scanner' ) ), absint( $core_deduped['count'] ) );
								?>
							</div>
						<?php endif; ?>
					</div>
				<?php endif; ?>

				<!-- Plugins Section -->
				<?php if ( ! empty( $scan_results['plugins']['vulnerable'] ) ) : ?>
					<div class="section">
						<h2 class="section-title">🔌 Plugins (<?php echo absint( $scan_results['plugins']['vulnerable'] ); ?>)</h2>
						<?php foreach ( $scan_results['plugins']['plugins'] as $plugin ) : ?>
							<?php if ( $plugin['vulnerable'] ) : ?>
								<?php $plugin_deduped = $this->deduplicate_vulnerabilities( $plugin['vulnerabilities'] ?? array() ); ?>
								<?php $plugin_total = count( $plugin_deduped['items'] ) + $plugin_deduped['count']; ?>
								<div class="item-card">
									<p class="item-name"><?php echo esc_html( $plugin['name'] ); ?></p>
									<p class="item-meta"><?php echo esc_html( $plugin['version'] ); ?> • <?php
										/* translators: %d: Number of issues */
										printf( esc_html( _n( '%d issue', '%d issues', $plugin_total, 'oopvulns-vulnerability-scanner' ) ), absint( $plugin_total ) );
									?></p>
									<?php foreach ( $plugin_deduped['items'] as $vuln ) : ?>
										<div class="vulnerability">
											<span class="vulnerability-name"><?php echo esc_html( $vuln['name'] ); ?></span>
											<span class="severity-badge"><?php echo esc_html( strtoupper( $vuln['severity'] ) ); ?></span>
										</div>
									<?php endforeach; ?>
									<?php if ( $plugin_deduped['count'] > 0 ) : ?>
										<div class="duplicate-note">
											+ <?php
											/* translators: %d: Number of duplicates */
											printf( esc_html( _n( '%d duplicate', '%d duplicates', $plugin_deduped['count'], 'oopvulns-vulnerability-scanner' ) ), absint( $plugin_deduped['count'] ) );
											?>
										</div>
									<?php endif; ?>
								</div>
							<?php endif; ?>
						<?php endforeach; ?>
					</div>
				<?php endif; ?>

				<!-- Themes Section -->
				<?php if ( ! empty( $scan_results['themes']['vulnerable'] ) ) : ?>
					<div class="section">
						<h2 class="section-title">🎨 Themes (<?php echo absint( $scan_results['themes']['vulnerable'] ); ?>)</h2>
						<?php foreach ( $scan_results['themes']['themes'] as $theme ) : ?>
							<?php if ( $theme['vulnerable'] ) : ?>
								<?php $theme_deduped = $this->deduplicate_vulnerabilities( $theme['vulnerabilities'] ?? array() ); ?>
								<?php $theme_total = count( $theme_deduped['items'] ) + $theme_deduped['count']; ?>
								<div class="item-card">
									<p class="item-name"><?php echo esc_html( $theme['name'] ); ?></p>
									<p class="item-meta"><?php echo esc_html( $theme['version'] ); ?> • <?php
										/* translators: %d: Number of issues */
										printf( esc_html( _n( '%d issue', '%d issues', $theme_total, 'oopvulns-vulnerability-scanner' ) ), absint( $theme_total ) );
									?></p>
									<?php foreach ( $theme_deduped['items'] as $vuln ) : ?>
										<div class="vulnerability">
											<span class="vulnerability-name"><?php echo esc_html( $vuln['name'] ); ?></span>
											<span class="severity-badge"><?php echo esc_html( strtoupper( $vuln['severity'] ) ); ?></span>
										</div>
									<?php endforeach; ?>
									<?php if ( $theme_deduped['count'] > 0 ) : ?>
										<div class="duplicate-note">
											+ <?php
											/* translators: %d: Number of duplicates */
											printf( esc_html( _n( '%d duplicate', '%d duplicates', $theme_deduped['count'], 'oopvulns-vulnerability-scanner' ) ), absint( $theme_deduped['count'] ) );
											?>
										</div>
									<?php endif; ?>
								</div>
							<?php endif; ?>
						<?php endforeach; ?>
					</div>
				<?php endif; ?>

				<!-- CTA -->
				<div style="text-align: center;">
					<a href="<?php echo esc_url( $admin_url ); ?>" class="cta-button"><?php esc_html_e( 'Review Full Report →', 'oopvulns-vulnerability-scanner' ); ?></a>
				</div>

			<?php else : ?>
				<!-- All Clear Section -->
				<div style="text-align: center; padding: 40px 0;">
					<div style="font-size: 64px; line-height: 1; margin-bottom: 16px;">✅</div>
					<h2 style="margin: 0 0 12px 0; font-size: 24px; font-weight: 700; color: #000000;"><?php esc_html_e( 'All Clear!', 'oopvulns-vulnerability-scanner' ); ?></h2>
					<p style="margin: 0; font-size: 14px; color: #666666;"><?php esc_html_e( 'No security vulnerabilities were detected on your WordPress site.', 'oopvulns-vulnerability-scanner' ); ?></p>
				</div>
			<?php endif; ?>
		</div>

		<!-- Footer -->
		<div class="footer">
			<p style="margin: 0 0 12px 0;">
				<strong><?php esc_html_e( 'OOPVulns Vulnerability Scanner', 'oopvulns-vulnerability-scanner' ); ?></strong>
			</p>
			<p style="margin: 0 0 12px 0; font-size: 11px;">
				<a href="<?php echo esc_url( $site_url ); ?>"><?php echo esc_html( $site_url ); ?></a>
			</p>
			<p style="margin: 0; font-size: 11px; color: #999999;">
				<?php esc_html_e( 'Keep your WordPress site secure with regular vulnerability scans.', 'oopvulns-vulnerability-scanner' ); ?>
			</p>
		</div>
	</div>
</body>
</html>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get email headers.
	 *
	 * @return array Email headers.
	 */
	private function get_headers() {
		return array(
			'Content-Type: text/html; charset=UTF-8',
			'From: ' . get_bloginfo( 'name' ) . ' <' . get_option( 'admin_email' ) . '>',
		);
	}
}
