<?php
/**
 * API Client Class
 *
 * @package OOPVulns
 */

namespace OOPVulns;

defined( 'ABSPATH' ) || exit;

/**
 * Handles all communication with the vulnerability API.
 */
class API_Client {
	/**
	 * API timeout in seconds.
	 *
	 * @var int
	 */
	private $timeout = 5;

	/**
	 * Get API key from available sources (constant > OOP Spam settings > OOPVulns settings).
	 *
	 * @return string API key or empty string if not found.
	 */
	private function get_api_key() {
		// Check for OOPSPAM_API_KEY constant first (highest priority).
		if ( defined( 'OOPSPAM_API_KEY' ) ) {
			$key = constant( 'OOPSPAM_API_KEY' );
			if ( ! empty( $key ) ) {
				return $key;
			}
		}

		// Check OOP Spam Anti-Spam plugin settings (second priority).
		$oopspam_settings = get_option( 'oopspamantispam_settings', array() );
		if ( ! empty( $oopspam_settings['oopspam_api_key'] ) ) {
			return $oopspam_settings['oopspam_api_key'];
		}

		// Check OOPVulns plugin settings (lowest priority).
		$oopvulns_settings = get_option( 'oopvulns_settings', array() );
		return $oopvulns_settings['api_key'] ?? '';
	}

	/**
	 * Get cache hours from settings.
	 *
	 * @return int
	 */
	private function get_cache_hours() {
		$settings = get_option( 'oopvulns_settings', array() );
		return isset( $settings['cache_hours'] ) ? absint( $settings['cache_hours'] ) : 12;
	}

	/**
	 * Make a GET request to the API.
	 *
	 * @param string $endpoint Endpoint path (e.g., 'core/6.4.2').
	 * @param bool   $use_cache Whether to use cache.
	 * @return array|false Response data or false on error.
	 */
	private function request( $endpoint, $use_cache = true ) {
		$cache_key = 'oopvulns_' . md5( $endpoint );

		// Try cache first.
		if ( $use_cache ) {
			$cached = get_transient( $cache_key );
			if ( false !== $cached ) {
				return $cached;
			}
		}

		// Make API request.
		$url      = OOPVULNS_API_BASE . $endpoint;
		$api_key  = $this->get_api_key();
		
		$headers = array(
			'Accept' => 'application/json',
		);
		
		if ( ! empty( $api_key ) ) {
			$headers['X-Api-Key'] = $api_key;
		}
		
		$response = wp_remote_get(
			$url,
			array(
				'timeout'    => $this->timeout,
				'user-agent' => $this->get_user_agent(),
				'headers'    => $headers,
			)
		);

		// Handle errors.
		if ( is_wp_error( $response ) ) {
			$this->log_error( 'API request failed', array(
				'endpoint' => $endpoint,
				'error'    => $response->get_error_message(),
			) );
			return false;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( $status_code !== 200 ) {
			$this->log_error( 'API returned non-200 status', array(
				'endpoint' => $endpoint,
				'status'   => $status_code,
			) );
			return false;
		}

		// Parse response.
		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->log_error( 'Failed to parse API response', array(
				'endpoint' => $endpoint,
				'error'    => json_last_error_msg(),
			) );
			return false;
		}

		// Validate response structure.
		if ( ! isset( $data['error'] ) ) {
			$this->log_error( 'Invalid API response structure', array(
				'endpoint' => $endpoint,
			) );
			return false;
		}

		// Cache successful responses.
		if ( $use_cache && ! $data['error'] ) {
			set_transient( $cache_key, $data, HOUR_IN_SECONDS * $this->get_cache_hours() );
		}

		return $data;
	}

	/**
	 * Get WordPress core vulnerabilities.
	 *
	 * @param string $version WordPress version.
	 * @param bool   $use_cache Whether to use cache.
	 * @return array|false Vulnerabilities or false.
	 */
	public function get_core_vulnerabilities( $version, $use_cache = true ) {
		$version = $this->sanitize_version( $version );
		if ( ! $version ) {
			return false;
		}
		
		$response = $this->request( "core/{$version}/", $use_cache );
		
		if ( ! $response || $response['error'] || empty( $response['data']['vulnerability'] ) ) {
			return false;
		}

		return $this->parse_core_vulnerabilities( $response['data']['vulnerability'] );
	}

	/**
	 * Get plugin vulnerabilities.
	 *
	 * @param string $slug Plugin slug.
	 * @param string $version Plugin version.
	 * @param bool   $use_cache Whether to use cache.
	 * @return array|false Vulnerabilities or false.
	 */
	public function get_plugin_vulnerabilities( $slug, $version, $use_cache = true ) {
		$slug = sanitize_title( $slug );
		if ( empty( $slug ) ) {
			return false;
		}

		$response = $this->request( "plugin/{$slug}/", $use_cache );
		
		if ( ! $response || $response['error'] || empty( $response['data']['vulnerability'] ) ) {
			return false;
		}

		return $this->parse_plugin_vulnerabilities( $response['data']['vulnerability'], $version );
	}

	/**
	 * Get theme vulnerabilities.
	 *
	 * @param string $slug Theme slug.
	 * @param string $version Theme version.
	 * @param bool   $use_cache Whether to use cache.
	 * @return array|false Vulnerabilities or false.
	 */
	public function get_theme_vulnerabilities( $slug, $version, $use_cache = true ) {
		$slug = sanitize_title( $slug );
		if ( empty( $slug ) ) {
			return false;
		}

		$response = $this->request( "theme/{$slug}/", $use_cache );
		
		if ( ! $response || $response['error'] || empty( $response['data']['vulnerability'] ) ) {
			return false;
		}

		return $this->parse_theme_vulnerabilities( $response['data']['vulnerability'], $version );
	}

	/**
	 * Parse core vulnerabilities from API response.
	 *
	 * @param array $vulnerabilities Raw vulnerability data.
	 * @return array Parsed vulnerabilities.
	 */
	private function parse_core_vulnerabilities( $vulnerabilities ) {
		$parsed = array();

		foreach ( $vulnerabilities as $vuln ) {
			$parsed[] = array(
				'name'        => sanitize_text_field( $vuln['name'] ?? '' ),
				'link'        => esc_url_raw( $vuln['link'] ?? '' ),
				'sources'     => $this->parse_sources( $vuln['source'] ?? array() ),
				'cvss_score'  => floatval( $vuln['impact']['cvss']['score'] ?? 0 ),
				'severity'    => sanitize_text_field( $vuln['impact']['cvss']['severity'] ?? 'unknown' ),
				'cwe'         => $this->parse_cwe( $vuln['impact']['cwe'] ?? array() ),
			);
		}

		return $parsed;
	}

	/**
	 * Parse plugin/theme vulnerabilities from API response.
	 *
	 * @param array  $vulnerabilities Raw vulnerability data.
	 * @param string $version Current version.
	 * @return array Parsed vulnerabilities.
	 */
	private function parse_plugin_vulnerabilities( $vulnerabilities, $version ) {
		$parsed = array();

		foreach ( $vulnerabilities as $vuln ) {
			$operator = $vuln['operator'] ?? array();
			
			// Check if vulnerability applies to this version.
			if ( ! $this->is_version_affected( $version, $operator ) ) {
				continue;
			}

			$parsed[] = array(
				'name'        => sanitize_text_field( $vuln['name'] ?? '' ),
				'description' => wp_kses_post( $vuln['description'] ?? '' ),
				'versions'    => $this->format_version_range( $operator ),
				'unfixed'     => ! empty( $operator['unfixed'] ),
				'closed'      => ! empty( $operator['closed'] ),
				'sources'     => $this->parse_sources( $vuln['source'] ?? array() ),
				'cvss_score'  => floatval( $vuln['impact']['cvss']['score'] ?? 0 ),
				'severity'    => sanitize_text_field( $vuln['impact']['cvss']['severity'] ?? 'unknown' ),
				'cwe'         => $this->parse_cwe( $vuln['impact']['cwe'] ?? array() ),
			);
		}

		return $parsed;
	}

	/**
	 * Parse theme vulnerabilities (same as plugin).
	 *
	 * @param array  $vulnerabilities Raw vulnerability data.
	 * @param string $version Current version.
	 * @return array Parsed vulnerabilities.
	 */
	private function parse_theme_vulnerabilities( $vulnerabilities, $version ) {
		return $this->parse_plugin_vulnerabilities( $vulnerabilities, $version );
	}

	/**
	 * Check if a version is affected by a vulnerability.
	 *
	 * @param string $version Version to check.
	 * @param array  $operator Operator constraints.
	 * @return bool True if affected.
	 */
	private function is_version_affected( $version, $operator ) {
		$version = $this->sanitize_version( $version );
		if ( ! $version ) {
			return false;
		}

		$min_op      = $operator['min_operator'] ?? '';
		$min_version = $operator['min_version'] ?? '';
		$max_op      = $operator['max_operator'] ?? '';
		$max_version = $operator['max_version'] ?? '';

		// Check minimum version constraint.
		if ( $min_op && $min_version ) {
			if ( ! version_compare( $version, $min_version, $min_op ) ) {
				return false;
			}
		}

		// Check maximum version constraint.
		if ( $max_op && $max_version ) {
			if ( ! version_compare( $version, $max_version, $max_op ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Format version range for display.
	 *
	 * @param array $operator Operator constraints.
	 * @return string Formatted range.
	 */
	public function format_version_range( $operator ) {
		$min_version = $operator['min_version'] ?? '';
		$max_version = $operator['max_version'] ?? '';
		$min_op = $operator['min_operator'] ?? '';
		$max_op = $operator['max_operator'] ?? '';

		// Single version constraint
		if ( $min_version && ! $max_version ) {
			return $this->format_operator( $min_op ) . ' ' . $min_version;
		}

		if ( $max_version && ! $min_version ) {
			return $this->format_operator( $max_op ) . ' ' . $max_version;
		}

		// Range: both min and max
		if ( $min_version && $max_version ) {
			// Common case: >= X.X and < Y.Y or <= Y.Y
			if ( ( $min_op === 'ge' || $min_op === 'gt' ) && ( $max_op === 'lt' || $max_op === 'le' ) ) {
				return sprintf( 'Versions %s - %s', $min_version, $max_version );
			}
			// Fallback for other combinations
			return sprintf( '%s %s, %s %s', $this->format_operator( $min_op ), $min_version, $this->format_operator( $max_op ), $max_version );
		}

		return 'All versions';
	}

	/**
	 * Convert operator code to human-readable text.
	 *
	 * @param string $op Operator code (lt, le, gt, ge, eq, ne).
	 * @return string Human-readable operator.
	 */
	public function format_operator( $op ) {
		$operators = array(
			'lt' => 'before',
			'le' => 'up to',
			'gt' => 'after',
			'ge' => 'from',
			'eq' => '',
			'ne' => 'not',
		);

		return $operators[ $op ] ?? $op;
	}

	/**
	 * Parse vulnerability sources.
	 *
	 * @param array $sources Raw sources.
	 * @return array Parsed sources.
	 */
	private function parse_sources( $sources ) {
		$parsed = array();

		foreach ( $sources as $source ) {
			$parsed[] = array(
				'name'        => sanitize_text_field( $source['name'] ?? '' ),
				'link'        => esc_url_raw( $source['link'] ?? '' ),
				'description' => sanitize_text_field( $source['description'] ?? '' ),
			);
		}

		return $parsed;
	}

	/**
	 * Parse CWE information.
	 *
	 * @param array $cwe_list Raw CWE data.
	 * @return array Parsed CWE data.
	 */
	private function parse_cwe( $cwe_list ) {
		$parsed = array();

		foreach ( $cwe_list as $cwe ) {
			$parsed[] = array(
				'name'        => sanitize_text_field( $cwe['name'] ?? '' ),
				'description' => sanitize_text_field( $cwe['description'] ?? '' ),
			);
		}

		return $parsed;
	}

	/**
	 * Sanitize version string.
	 *
	 * @param string $version Version string.
	 * @return string|false Sanitized version or false.
	 */
	private function sanitize_version( $version ) {
		$version = preg_replace( '/[^0-9.]/', '', $version );
		return ! empty( $version ) ? $version : false;
	}

	/**
	 * Get user agent string.
	 *
	 * @return string User agent.
	 */
	private function get_user_agent() {
		return sprintf(
			'OOPVulns/%s (WordPress/%s; %s)',
			OOPVULNS_VERSION,
			get_bloginfo( 'version' ),
			home_url()
		);
	}

	/**
	 * Log error message.
	 *
	 * @param string $message Error message.
	 * @param array  $context Additional context.
	 */
	private function log_error( $message, $context = array() ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			$log_message = sprintf(
				'[ OOPVulns ] %s: %s',
				$message,
				wp_json_encode( $context )
			);
			do_action( 'oopvulns_debug_log', $log_message, $message, $context );
		}
	}
}
