<?php
/**
 * Admin Class
 *
 * @package OOPVulns
 */

namespace OOPVulns;

defined( 'ABSPATH' ) || exit;

/**
 * Handles all admin functionality and UI.
 */
class Admin {
	/**
	 * Scanner instance.
	 *
	 * @var Scanner
	 */
	private $scanner;

	/**
	 * Notifications instance.
	 *
	 * @var Notifications
	 */
	private $notifications;

	/**
	 * Constructor.
	 *
	 * @param Scanner       $scanner Scanner instance.
	 * @param Notifications $notifications Notifications instance.
	 */
	public function __construct( Scanner $scanner, Notifications $notifications ) {
		$this->scanner       = $scanner;
		$this->notifications = $notifications;

		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_init', array( $this, 'handle_actions' ) );
		add_action( 'admin_notices', array( $this, 'show_admin_notices' ) );

		// Add vulnerability notices on plugins and themes pages.
		add_action( 'admin_head', array( $this, 'register_plugin_vulnerability_notices' ) );
		add_action( 'admin_head', array( $this, 'register_theme_vulnerability_notices' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_vulnerability_page_styles' ) );

		// AJAX handlers for scanning with progress.
		add_action( 'wp_ajax_oopvulns_start_scan', array( $this, 'ajax_start_scan' ) );
		add_action( 'wp_ajax_oopvulns_scan_item', array( $this, 'ajax_scan_item' ) );
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		add_management_page(
			__( ' OOPVulns ', 'oopvulns-vulnerability-scanner' ),
			__( 'Vulnerability Scanner', 'oopvulns-vulnerability-scanner' ),
			'manage_options',
			'oopvulns-vulnerability-scanner',
			array( $this, 'render_admin_page' )
		);
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_scripts( $hook ) {
		if ( 'tools_page_oopvulns-vulnerability-scanner' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'oopvulns-admin',
			OOPVULNS_URL . 'assets/css/admin.css',
			array(),
			OOPVULNS_VERSION
		);

		wp_enqueue_script(
			'oopvulns-admin',
			OOPVULNS_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			OOPVULNS_VERSION,
			true
		);

		wp_localize_script(
			'oopvulns-admin',
			'oopvulnsData',
			array(
				'ajaxUrl'        => admin_url( 'admin-ajax.php' ),
				'nonce'          => wp_create_nonce( 'oopvulns_ajax' ),
				'i18n'           => array(
					'scanning'     => __( 'Scanning...', 'oopvulns-vulnerability-scanner' ),
					'scanComplete' => __( 'Scan complete!', 'oopvulns-vulnerability-scanner' ),
					'scanError'    => __( 'Scan failed. Please try again.', 'oopvulns-vulnerability-scanner' ),
				),
			)
		);
	}

	/**
	 * Get the active API key and its source.
	 *
	 * @return array Array with 'key' and 'source' keys.
	 */
	private function get_api_key_info() {
		// Check for OOPSPAM_API_KEY constant first (highest priority).
		if ( defined( 'OOPSPAM_API_KEY' ) ) {
			$key = constant( 'OOPSPAM_API_KEY' );
			if ( ! empty( $key ) ) {
				return array(
					'key'    => $key,
					'source' => 'constant',
				);
			}
		}

		// Check OOP Spam Anti-Spam plugin settings (second priority).
		$oopspam_settings = get_option( 'oopspamantispam_settings', array() );
		if ( ! empty( $oopspam_settings['oopspam_api_key'] ) ) {
			return array(
				'key'    => $oopspam_settings['oopspam_api_key'],
				'source' => 'oopspam',
			);
		}

		// Check OOPVulns plugin settings (lowest priority).
		$oopvulns_settings = get_option( 'oopvulns_settings', array() );
		$key               = $oopvulns_settings['api_key'] ?? '';
		return array(
			'key'    => $key,
			'source' => 'local',
		);
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		register_setting(
			'oopvulns_settings',
			'oopvulns_settings',
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
			)
		);

		// Update schedules when settings are saved.
		add_action( 'update_option_oopvulns_settings', array( $this, 'update_schedules_on_save' ), 10, 2 );
	}

	/**
	 * Update schedules when settings are saved.
	 *
	 * @param array $old_value Old settings value.
	 * @param array $new_value New settings value.
	 */
	public function update_schedules_on_save( $old_value, $new_value ) {
		// Check if scan frequency or notification settings changed.
		$scan_changed = ( $old_value['scan_frequency'] ?? '' ) !== ( $new_value['scan_frequency'] ?? '' );
		$notification_changed = ( $old_value['notification_schedule'] ?? '' ) !== ( $new_value['notification_schedule'] ?? '' ) ||
		                       ( $old_value['notification_day'] ?? '' ) !== ( $new_value['notification_day'] ?? '' ) ||
		                       ( $old_value['notification_hour'] ?? '' ) !== ( $new_value['notification_hour'] ?? '' ) ||
		                       ( $old_value['enabled'] ?? '' ) !== ( $new_value['enabled'] ?? '' );

		// Trigger schedule updates.
		if ( $scan_changed || $notification_changed ) {
			do_action( 'oopvulns_schedules_update' );
		}
	}

	/**
	 * Sanitize settings.
	 *
	 * @param array $input Raw input.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		// API Key
		$sanitized['api_key'] = sanitize_text_field( $input['api_key'] ?? '' );

		$sanitized['cache_hours'] = absint( $input['cache_hours'] ?? 12 );
		if ( ! in_array( $sanitized['cache_hours'], array( 1, 6, 12, 24 ), true ) ) {
			$sanitized['cache_hours'] = 12;
		}

		$sanitized['scan_frequency'] = sanitize_text_field( $input['scan_frequency'] ?? 'daily' );
		if ( ! in_array( $sanitized['scan_frequency'], array( 'daily', 'weekly' ), true ) ) {
			$sanitized['scan_frequency'] = 'daily';
		}

		$sanitized['notification_email'] = sanitize_email( $input['notification_email'] ?? get_option( 'admin_email' ) );
		if ( empty( $sanitized['notification_email'] ) ) {
			$sanitized['notification_email'] = get_option( 'admin_email' );
		}
		
		$sanitized['notification_schedule'] = sanitize_text_field( $input['notification_schedule'] ?? 'weekly' );
		if ( ! in_array( $sanitized['notification_schedule'], array( 'daily', 'weekly', 'never' ), true ) ) {
			$sanitized['notification_schedule'] = 'weekly';
		}

		$sanitized['notification_day'] = sanitize_text_field( $input['notification_day'] ?? 'monday' );
		$valid_days = array( 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday' );
		if ( ! in_array( $sanitized['notification_day'], $valid_days, true ) ) {
			$sanitized['notification_day'] = 'monday';
		}

		$sanitized['notification_hour'] = absint( $input['notification_hour'] ?? 9 );
		if ( $sanitized['notification_hour'] > 23 ) {
			$sanitized['notification_hour'] = 9;
		}

		$sanitized['notification_minute'] = absint( $input['notification_minute'] ?? 0 );
		if ( $sanitized['notification_minute'] > 59 ) {
			$sanitized['notification_minute'] = 0;
		}

		$sanitized['enabled'] = ! empty( $input['enabled'] );

		return $sanitized;
	}

	/**
	 * Handle admin actions.
	 */
	public function handle_actions() {
		if ( ! isset( $_POST['oopvulns_action'] ) || ! current_user_can( 'manage_options' ) ) {
			return;
		}

		check_admin_referer( 'oopvulns_action' );

		$action = sanitize_text_field( wp_unslash( $_POST['oopvulns_action'] ) );

		switch ( $action ) {
			case 'scan_now':
				$this->scanner->scan_all( false );
				$this->scanner->update_last_scan_time();
				add_settings_error(
					'oopvulns_messages',
					'scan_complete',
					__( 'Vulnerability scan completed successfully!', 'oopvulns-vulnerability-scanner' ),
					'success'
				);
				break;

			case 'send_test_email':
				$results = array(
					'core'    => get_option( 'oopvulns_core_result', array() ),
					'plugins' => get_option( 'oopvulns_plugins_result', array() ),
					'themes'  => get_option( 'oopvulns_themes_result', array() ),
					'summary' => $this->scanner->get_scan_summary(),
				);
				$sent = $this->notifications->send_notification( $results, true );
				
				if ( $sent ) {
					add_settings_error(
						'oopvulns_messages',
						'email_sent',
						__( 'Test email sent successfully!', 'oopvulns-vulnerability-scanner' ),
						'success'
					);
				} else {
					add_settings_error(
						'oopvulns_messages',
						'email_failed',
						__( 'Failed to send test email. Please check your email settings.', 'oopvulns-vulnerability-scanner' ),
						'error'
					);
				}
				break;

			case 'clear_cache':
				$this->scanner->clear_results();
				add_settings_error(
					'oopvulns_messages',
					'cache_cleared',
					__( 'Cache cleared successfully!', 'oopvulns-vulnerability-scanner' ),
					'success'
				);
				break;
		}

		set_transient( 'oopvulns_messages', get_settings_errors( 'oopvulns_messages' ), 30 );
		
		wp_safe_redirect( add_query_arg( 'page', 'oopvulns-vulnerability-scanner', admin_url( 'tools.php' ) ) );
		exit;
	}

	/**
	 * Show admin notices.
	 */
	public function show_admin_notices() {
		$screen = get_current_screen();
		if ( ! $screen ) {
			return;
		}

		// Show vulnerability warning on Themes page.
		if ( 'themes' === $screen->id ) {
			$themes_result = get_option( 'oopvulns_themes_result', array() );
			
			if ( ! empty( $themes_result['vulnerable'] ) && $themes_result['vulnerable'] > 0 ) {
				$message = sprintf(
					/* translators: %s: Number of vulnerable themes */
					_n(
						'<strong>Security Warning:</strong>  OOPVulns  found %d vulnerable theme installed on your site.',
						'<strong>Security Warning:</strong>  OOPVulns  found %d vulnerable themes installed on your site.',
						$themes_result['vulnerable'],
						'oopvulns-vulnerability-scanner'
					),
					$themes_result['vulnerable']
				);
				
				$link = admin_url( 'tools.php?page=oopvulns-vulnerability-scanner' );
				$message .= sprintf(
					' <a href="%s">%s</a>',
					esc_url( $link ),
					esc_html__( 'View Vulnerability Report', 'oopvulns-vulnerability-scanner' )
				);

				printf(
					'<div class="notice notice-error"><p>%s</p></div>',
					wp_kses_post( $message )
				);
			}
		}

		if ( 'tools_page_oopvulns-vulnerability-scanner' !== $screen->id ) {
			return;
		}

		$messages = get_transient( 'oopvulns_messages' );
		if ( ! $messages ) {
			return;
		}

		delete_transient( 'oopvulns_messages' );

		foreach ( $messages as $message ) {
			printf(
				'<div class="notice notice-%s is-dismissible"><p>%s</p></div>',
				esc_attr( $message['type'] ),
				esc_html( $message['message'] )
			);
		}
	}

	/**
	 * Format version string into human-readable text.
	 *
	 * @param string $versions Raw version string from vulnerability data.
	 * @return string Formatted version string.
	 */
	private function format_versions( $versions ) {
		if ( empty( $versions ) ) {
			return '';
		}

		// Replace operator symbols with human-readable text
		$formatted = preg_replace_callback(
			'/(<=|<|>=|>|=)\s*([^\s,]+)/',
			function( $matches ) {
				$operator = trim( $matches[1] );
				$version  = trim( $matches[2] );

				switch ( $operator ) {
					case '<=':
						return 'up to ' . $version;
					case '<':
						return 'before ' . $version;
					case '>=':
						return 'from ' . $version;
					case '>':
						return 'after ' . $version;
					case '=':
						return 'version ' . $version;
					default:
						return $matches[0];
				}
			},
			$versions
		);

		// Clean up wildcards and extra commas
		$formatted = str_replace( array( ' *', ',', '  ' ), array( '', ', ', ' ' ), $formatted );
		return trim( $formatted );
	}

	/**
	 * Deduplicate vulnerabilities by generic product name + version constraint.
	 * Handles variations like "Product [slug] <= 1.3.0 (unfixed)" and "Product <= 1.3.0 - Missing Auth"
	 * Returns both unique items and duplicate items (for optional unhiding).
	 *
	 * @param array $vulnerabilities Vulnerability list.
	 * @return array{items: array, duplicates: array, count: int}
	 */
	private function deduplicate_vulnerabilities( $vulnerabilities ) {
		if ( empty( $vulnerabilities ) || ! is_array( $vulnerabilities ) ) {
			return array(
				'items'      => array(),
				'duplicates' => array(),
				'count'      => 0,
			);
		}

		$items            = array();
		$duplicates       = array();
		$seen             = array();

		foreach ( $vulnerabilities as $vuln ) {
			$name = strtolower( trim( (string) ( $vuln['name'] ?? '' ) ) );

			// Remove bracketed content [like-this]
			$clean_name = preg_replace( '/\s*\[[^\]]*\]/', '', $name );

			// Extract version constraint (e.g., "<=", ">=", "=", etc. followed by version)
			$version_constraint = '';
			if ( preg_match( '/(<=|>=|<|>|=)\s*(\d+[\.\d]*)/', $clean_name, $matches ) ) {
				$version_constraint = $matches[1] . ' ' . $matches[2];
			}

			// Remove everything after typical description separators (-, (,  etc)
			// Keep only the product name and version constraint
			$product_part = preg_replace( '/\s*[-–]\s*.*/i', '', $clean_name );
			$product_part = preg_replace( '/\s*\(.*/i', '', $product_part );
			
			// Extract just the product name before version constraint
			$product_part = preg_replace( '/(<=|>=|<|>|=).*/i', '', $product_part );
			$product_name = strtolower( trim( $product_part ) );

			// Create dedup key: product name + version constraint
			$key = $product_name . '|' . $version_constraint;

			if ( isset( $seen[ $key ] ) ) {
				$duplicates[] = $vuln;
				continue;
			}

			$seen[ $key ] = true;
			$items[]      = $vuln;
		}

		return array(
			'items'      => $items,
			'duplicates' => $duplicates,
			'count'      => count( $duplicates ),
		);
	}

	/**
	 * Render admin page.
	 */
	public function render_admin_page() {
		$active_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'dashboard';
		$settings   = get_option( 'oopvulns_settings', array() );
		$summary    = $this->scanner->get_scan_summary();
		?>
		<div class="wrap oopvulns-admin">
			<h1 class="oopvulns-title">
				<span class="dashicons dashicons-shield-alt" aria-hidden="true"></span>
				<?php esc_html_e( ' OOPVulns ', 'oopvulns-vulnerability-scanner' ); ?>
			</h1>

			<nav class="nav-tab-wrapper wp-clearfix" role="tablist" aria-label="<?php esc_attr_e( 'Plugin sections', 'oopvulns-vulnerability-scanner' ); ?>">
				<a href="<?php echo esc_url( add_query_arg( 'tab', 'dashboard', admin_url( 'tools.php?page=oopvulns-vulnerability-scanner' ) ) ); ?>" 
				   class="nav-tab <?php echo $active_tab === 'dashboard' ? 'nav-tab-active' : ''; ?>"
				   role="tab"
				   aria-selected="<?php echo $active_tab === 'dashboard' ? 'true' : 'false'; ?>"
				   aria-controls="dashboard-panel">
					<?php esc_html_e( 'Dashboard', 'oopvulns-vulnerability-scanner' ); ?>
				</a>
				<a href="<?php echo esc_url( add_query_arg( 'tab', 'settings', admin_url( 'tools.php?page=oopvulns-vulnerability-scanner' ) ) ); ?>" 
				   class="nav-tab <?php echo $active_tab === 'settings' ? 'nav-tab-active' : ''; ?>"
				   role="tab"
				   aria-selected="<?php echo $active_tab === 'settings' ? 'true' : 'false'; ?>"
				   aria-controls="settings-panel">
					<?php esc_html_e( 'Settings', 'oopvulns-vulnerability-scanner' ); ?>
				</a>
			</nav>

			<div class="oopvulns-content">
				<?php
				switch ( $active_tab ) {
					case 'dashboard':
						$this->render_dashboard_tab( $summary );
						break;
					case 'settings':
						$this->render_settings_tab( $settings );
						break;
				}
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render dashboard tab.
	 *
	 * @param array $summary Scan summary.
	 */
	private function render_dashboard_tab( $summary ) {
		$core_result    = get_option( 'oopvulns_core_result', array() );
		$plugins_result = get_option( 'oopvulns_plugins_result', array() );
		$themes_result  = get_option( 'oopvulns_themes_result', array() );
		$last_scan      = $summary['last_scan'] ?? false;
		$total_vulns    = $summary['total_vulnerabilities'] ?? 0;
		$has_vulns      = $total_vulns > 0;
		?>
		<div id="dashboard-panel" role="tabpanel" aria-labelledby="dashboard-tab" class="oopvulns-panel">
			
			<!-- Status Hero -->
			<div class="oopvulns-status-hero <?php echo $has_vulns ? 'has-vulnerabilities' : 'is-secure'; ?>">
				<div class="oopvulns-status-icon">
					<span class="dashicons <?php echo $has_vulns ? 'dashicons-warning' : 'dashicons-yes-alt'; ?>" aria-hidden="true"></span>
				</div>
				<h2 class="oopvulns-status-title">
					<?php echo $has_vulns ? esc_html__( 'Vulnerabilities Found', 'oopvulns-vulnerability-scanner' ) : esc_html__( 'No Known Vulnerabilities', 'oopvulns-vulnerability-scanner' ); ?>
				</h2>
				<p class="oopvulns-status-subtitle">
					<?php echo $has_vulns 
						? esc_html__( 'Security issues detected that require your attention', 'oopvulns-vulnerability-scanner' )
						: esc_html__( 'No known vulnerabilities in your WordPress installation', 'oopvulns-vulnerability-scanner' ); ?>
				</p>
				<div class="oopvulns-status-count"><?php echo absint( $total_vulns ); ?></div>
				<div class="oopvulns-status-label">
					<?php echo esc_html( _n( 'Vulnerability', 'Vulnerabilities', $total_vulns, 'oopvulns-vulnerability-scanner' ) ); ?>
				</div>
			</div>

			<!-- Quick Stats -->
			<div class="oopvulns-quick-stats">
				<div class="oopvulns-stat-item <?php echo ! empty( $summary['core_vulnerable'] ) ? 'has-issues' : ''; ?>">
					<div class="oopvulns-stat-icon">
						<span class="dashicons dashicons-wordpress" aria-hidden="true"></span>
					</div>
					<div class="oopvulns-stat-info">
						<div class="oopvulns-stat-value">
							<?php echo ! empty( $summary['core_vulnerable'] ) ? esc_html__( 'Vulnerable', 'oopvulns-vulnerability-scanner' ) : esc_html__( 'No known vulnerabilities', 'oopvulns-vulnerability-scanner' ); ?>
						</div>
						<div class="oopvulns-stat-name"><?php esc_html_e( 'WordPress Core', 'oopvulns-vulnerability-scanner' ); ?></div>
					</div>
				</div>
				<div class="oopvulns-stat-item <?php echo ! empty( $summary['vulnerable_plugins'] ) ? 'has-issues' : ''; ?>">
					<div class="oopvulns-stat-icon">
						<span class="dashicons dashicons-admin-plugins" aria-hidden="true"></span>
					</div>
					<div class="oopvulns-stat-info">
						<div class="oopvulns-stat-value">
							<?php echo absint( $summary['vulnerable_plugins'] ?? 0 ); ?>
						</div>
						<div class="oopvulns-stat-name"><?php esc_html_e( 'Vulnerable Plugins', 'oopvulns-vulnerability-scanner' ); ?></div>
					</div>
				</div>
				<div class="oopvulns-stat-item <?php echo ! empty( $summary['vulnerable_themes'] ) ? 'has-issues' : ''; ?>">
					<div class="oopvulns-stat-icon">
						<span class="dashicons dashicons-admin-appearance" aria-hidden="true"></span>
					</div>
					<div class="oopvulns-stat-info">
						<div class="oopvulns-stat-value">
							<?php echo absint( $summary['vulnerable_themes'] ?? 0 ); ?>
						</div>
						<div class="oopvulns-stat-name"><?php esc_html_e( 'Vulnerable Themes', 'oopvulns-vulnerability-scanner' ); ?></div>
					</div>
				</div>
			</div>

			<!-- Actions -->
			<div class="oopvulns-actions">
				<form id="oopvulns-scan-form" method="post" action="">
					<!-- Progress Bar (hidden by default) -->
					<div class="oopvulns-scan-progress" id="oopvulns-scan-progress" style="display: none;">
						<div class="oopvulns-progress-header">
							<span class="oopvulns-progress-text" id="oopvulns-progress-text"><?php esc_html_e( 'Initializing scan...', 'oopvulns-vulnerability-scanner' ); ?></span>
							<span class="oopvulns-progress-count" id="oopvulns-progress-count">0 / 0</span>
						</div>
						<div class="oopvulns-progress-bar-container">
							<div class="oopvulns-progress-bar" id="oopvulns-progress-bar" style="width: 0%;"></div>
						</div>
						<div class="oopvulns-progress-item" id="oopvulns-progress-item"></div>
					</div>

					<div style="display: flex; align-items: center; gap: 16px; flex-wrap: wrap; justify-content: center;">
						<input type="hidden" name="oopvulns_action" value="scan_now">
						<button type="submit" class="oopvulns-scan-btn" id="oopvulns-scan-btn">
							<span class="dashicons dashicons-update" aria-hidden="true"></span>
							<?php esc_html_e( 'Run Scan Now', 'oopvulns-vulnerability-scanner' ); ?>
						</button>

						<div class="oopvulns-scan-options" style="display: flex; gap: 8px; align-items: center; flex-wrap: wrap;">
							<label for="scan-type" style="font-weight: 600; margin: 0;"><?php esc_html_e( 'Scan:', 'oopvulns-vulnerability-scanner' ); ?></label>
							<select id="scan-type" name="scan_type" style="padding: 6px 8px; border-radius: 4px; border: 1px solid #8c8f94;">
								<option value="all"><?php esc_html_e( 'All (Core, Plugins & Themes)', 'oopvulns-vulnerability-scanner' ); ?></option>
								<option value="core"><?php esc_html_e( 'Core Only', 'oopvulns-vulnerability-scanner' ); ?></option>
								<option value="plugins"><?php esc_html_e( 'Plugins Only', 'oopvulns-vulnerability-scanner' ); ?></option>
								<option value="themes"><?php esc_html_e( 'Themes Only', 'oopvulns-vulnerability-scanner' ); ?></option>
							</select>
						</div>
					</div>
				</form>

				<?php if ( $last_scan ) : ?>
					<span class="oopvulns-last-scan">
						<?php
						printf(
							/* translators: %s: Time since last scan */
							esc_html__( 'Last scan: %s ago', 'oopvulns-vulnerability-scanner' ),
							esc_html( human_time_diff( $last_scan, current_time( 'timestamp' ) ) )
						);
						?>
					</span>
				<?php endif; ?>
			</div>

			<!-- Vulnerability Details -->
			<?php
			$this->render_core_vulnerabilities( $core_result );
			$this->render_plugin_vulnerabilities( $plugins_result );
			$this->render_theme_vulnerabilities( $themes_result );
			?>
		</div>
		<?php
	}

	/**
	 * Render core vulnerabilities section.
	 *
	 * @param array $core_result Core scan result.
	 */
	private function render_core_vulnerabilities( $core_result ) {
		if ( empty( $core_result['vulnerable'] ) ) {
			return;
		}

		$deduped_core_vulns    = $this->deduplicate_vulnerabilities( $core_result['vulnerabilities'] ?? array() );
		$core_vulnerabilities  = $deduped_core_vulns['items'];
		$core_duplicates       = $deduped_core_vulns['duplicates'];
		$core_duplicates_count = $deduped_core_vulns['count'];
		?>
		<div class="oopvulns-section">
			<h2 class="oopvulns-section-title">
				<span class="dashicons dashicons-wordpress" aria-hidden="true"></span>
				<?php esc_html_e( 'WordPress Core Vulnerabilities', 'oopvulns-vulnerability-scanner' ); ?>
			</h2>

			<div class="oopvulns-card oopvulns-card-danger">
				<div class="oopvulns-card-body">
					<?php if ( $core_duplicates_count > 0 ) : ?>
						<div class="oopvulns-duplicates-notice">
							<p class="description">
								<span class="oopvulns-duplicates-count">
									<?php
									printf(
										/* translators: %d: number of duplicate vulnerabilities hidden */
										esc_html__( '%d possible duplicate vulnerability entries were hidden.', 'oopvulns-vulnerability-scanner' ),
										absint( $core_duplicates_count )
									);
									?>
								</span>
								<button type="button" class="oopvulns-toggle-duplicates" data-section="wordpress-core">
									<span class="oopvulns-toggle-show"><?php esc_html_e( 'Show', 'oopvulns-vulnerability-scanner' ); ?></span>
									<span class="oopvulns-toggle-hide" style="display: none;"><?php esc_html_e( 'Hide', 'oopvulns-vulnerability-scanner' ); ?></span>
								</button>
							</p>
							<ul class="oopvulns-inline-vuln-list oopvulns-hidden-duplicates" id="wordpress-core-duplicates" style="display: none; margin-top: 12px;">
								<?php foreach ( $core_duplicates as $dup_vuln ) : ?>
									<li class="oopvulns-inline-vuln-list-item oopvulns-duplicate-highlight">
										<span class="oopvulns-inline-severity oopvulns-severity-<?php echo esc_attr( strtolower( $dup_vuln['severity'] ?? 'low' ) ); ?>">
											<?php echo esc_html( ucfirst( $dup_vuln['severity'] ?? 'Unknown' ) ); ?>
										</span>
										<span class="oopvulns-inline-vuln-name"><?php echo esc_html( $dup_vuln['name'] ?? 'Security Vulnerability' ); ?></span>
										<span class="description" style="margin-left: 8px;"><?php esc_html_e( '(Duplicate)', 'oopvulns-vulnerability-scanner' ); ?></span>
									</li>
								<?php endforeach; ?>
							</ul>
						</div>
					<?php endif; ?>
					<p class="oopvulns-alert-text">
						<?php
						printf(
							/* translators: %s: WordPress version */
							esc_html__( 'Your WordPress version (%s) has known security vulnerabilities.', 'oopvulns-vulnerability-scanner' ),
							'<strong>' . esc_html( $core_result['version'] ) . '</strong>'
						);
						?>
					</p>

					<?php foreach ( $core_vulnerabilities as $vuln ) : ?>
						<div class="oopvulns-vulnerability">
							<div class="oopvulns-vuln-header">
								<h3 class="oopvulns-vuln-title"><?php echo esc_html( $vuln['name'] ); ?></h3>
								<?php $this->render_severity_badge( $vuln['severity'], $vuln['cvss_score'] ); ?>
							</div>

							<?php if ( ! empty( $vuln['cwe'] ) ) : ?>
								<div class="oopvulns-cwe-list">
									<?php foreach ( $vuln['cwe'] as $cwe ) : ?>
										<div class="oopvulns-cwe-item">
											<strong><?php echo esc_html( $cwe['name'] ); ?>:</strong>
											<?php echo esc_html( $cwe['description'] ); ?>
										</div>
									<?php endforeach; ?>
								</div>
							<?php endif; ?>

							<?php if ( ! empty( $vuln['sources'] ) ) : ?>
								<div class="oopvulns-sources">
									<strong><?php esc_html_e( 'References:', 'oopvulns-vulnerability-scanner' ); ?></strong>
									<ul>
										<?php foreach ( $vuln['sources'] as $source ) : ?>
											<li>
												<a href="<?php echo esc_url( $source['link'] ); ?>" 
												   target="_blank" 
												   rel="noopener noreferrer">
													<?php echo esc_html( $source['name'] ); ?>
													<span class="screen-reader-text"><?php esc_html_e( '(opens in new tab)', 'oopvulns-vulnerability-scanner' ); ?></span>
												</a>
											</li>
										<?php endforeach; ?>
									</ul>
								</div>
							<?php endif; ?>

							<div class="oopvulns-action">
								<a href="<?php echo esc_url( admin_url( 'update-core.php' ) ); ?>" class="button button-primary">
									<?php esc_html_e( 'Update WordPress', 'oopvulns-vulnerability-scanner' ); ?>
								</a>
							</div>
						</div>
					<?php endforeach; ?>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render plugin vulnerabilities section.
	 *
	 * @param array $plugins_result Plugins scan result.
	 */
	private function render_plugin_vulnerabilities( $plugins_result ) {
		if ( empty( $plugins_result['vulnerable'] ) ) {
			return;
		}
		?>
		<div class="oopvulns-section">
			<h2 class="oopvulns-section-title">
				<span class="dashicons dashicons-admin-plugins" aria-hidden="true"></span>
				<?php esc_html_e( 'Plugin Vulnerabilities', 'oopvulns-vulnerability-scanner' ); ?>
				<span class="oopvulns-section-badge"><?php echo absint( $plugins_result['vulnerable'] ); ?></span>
			</h2>

			<?php foreach ( $plugins_result['plugins'] as $plugin ) : ?>
				<?php if ( ! $plugin['vulnerable'] ) continue; ?>
				<?php $card_id = 'plugin-' . sanitize_html_class( $plugin['slug'] ); ?>
				<?php
				$deduped_plugin_vulns      = $this->deduplicate_vulnerabilities( $plugin['vulnerabilities'] ?? array() );
				$plugin_vulnerabilities    = $deduped_plugin_vulns['items'];
				$plugin_duplicates         = $deduped_plugin_vulns['duplicates'];
				$plugin_duplicates_count   = $deduped_plugin_vulns['count'];
				?>

				<div class="oopvulns-card oopvulns-card-warning oopvulns-collapsible" data-card-id="<?php echo esc_attr( $card_id ); ?>">
					<button type="button" class="oopvulns-card-header" aria-expanded="false" aria-controls="<?php echo esc_attr( $card_id ); ?>-body">
						<h3>
							<?php echo esc_html( $plugin['name'] ); ?>
							<span class="oopvulns-version-badge">v<?php echo esc_html( $plugin['version'] ); ?></span>
							<?php if ( ! empty( $plugin['active'] ) ) : ?>
								<span class="oopvulns-badge oopvulns-badge-info"><?php esc_html_e( 'Active', 'oopvulns-vulnerability-scanner' ); ?></span>
							<?php endif; ?>
						</h3>
						<span class="oopvulns-card-toggle">
							<span class="dashicons dashicons-arrow-down-alt2" aria-hidden="true"></span>
						</span>
					</button>

					<div class="oopvulns-card-body" id="<?php echo esc_attr( $card_id ); ?>-body" hidden>
						<?php if ( $plugin_duplicates_count > 0 ) : ?>
							<div class="oopvulns-duplicates-notice">
								<p class="description">
									<span class="oopvulns-duplicates-count">
										<?php
										printf(
											/* translators: %d: number of duplicate vulnerabilities hidden */
											esc_html__( '%d possible duplicate vulnerability entries were hidden.', 'oopvulns-vulnerability-scanner' ),
											absint( $plugin_duplicates_count )
										);
										?>
									</span>
									<button type="button" class="oopvulns-toggle-duplicates" data-section="<?php echo esc_attr( $card_id ); ?>">
										<span class="oopvulns-toggle-show"><?php esc_html_e( 'Show', 'oopvulns-vulnerability-scanner' ); ?></span>
										<span class="oopvulns-toggle-hide" style="display: none;"><?php esc_html_e( 'Hide', 'oopvulns-vulnerability-scanner' ); ?></span>
									</button>
								</p>
								<div class="oopvulns-hidden-duplicates" id="<?php echo esc_attr( $card_id ); ?>-duplicates" style="display: none;">
									<?php foreach ( $plugin_duplicates as $dup_vuln ) : ?>
										<div class="oopvulns-vulnerability oopvulns-duplicate-highlight">
											<div class="oopvulns-vuln-header">
												<h4 class="oopvulns-vuln-title"><?php echo esc_html( $dup_vuln['name'] ); ?></h4>
												<?php $this->render_severity_badge( $dup_vuln['severity'], $dup_vuln['cvss_score'] ); ?>
											</div>
											<p class="description" style="margin-top: 8px; font-style: italic;"><?php esc_html_e( '(Duplicate)', 'oopvulns-vulnerability-scanner' ); ?></p>
										</div>
									<?php endforeach; ?>
								</div>
							</div>
						<?php endif; ?>
						<?php foreach ( $plugin_vulnerabilities as $vuln ) : ?>
							<div class="oopvulns-vulnerability">
								<div class="oopvulns-vuln-header">
									<h4 class="oopvulns-vuln-title"><?php echo esc_html( $vuln['name'] ); ?></h4>
									<?php $this->render_severity_badge( $vuln['severity'], $vuln['cvss_score'] ); ?>
								</div>

								<?php if ( ! empty( $vuln['description'] ) ) : ?>
									<p class="oopvulns-vuln-description"><?php echo wp_kses_post( $vuln['description'] ); ?></p>
								<?php endif; ?>

								<?php if ( ! empty( $vuln['unfixed'] ) ) : ?>
						<?php
						// Check if an update is available for this plugin
						$update_available = false;
						$update_plugins = get_site_transient( 'update_plugins' );
						if ( ! empty( $update_plugins->response[ $plugin['file'] ] ) ) {
							$update_available = true;
						}
						?>
						<div class="oopvulns-alert oopvulns-alert-danger">
							<div style="display: flex; align-items: center; gap: 8px;">
								<span class="dashicons dashicons-warning" aria-hidden="true"></span>
								<span>
									<?php if ( $update_available ) : ?>
											<?php esc_html_e( 'Potential unpatched vulnerability detected. Please update this plugin to the latest version first, then re-scan.', 'oopvulns-vulnerability-scanner' ); ?>
									<?php else : ?>
											<?php esc_html_e( 'Potential unpatched vulnerability detected. Please verify you are on the latest version and re-scan. Vulnerability data can occasionally lag behind new releases.', 'oopvulns-vulnerability-scanner' ); ?>
									<?php endif; ?>
								</span>
							</div>
						</div>
								<?php endif; ?>
								
								<p class="oopvulns-affected-versions">
									<strong><?php esc_html_e( 'Affected:', 'oopvulns-vulnerability-scanner' ); ?></strong>
									<?php echo esc_html( $this->format_versions( $vuln['versions'] ) ); ?>
								</p>

								<?php if ( ! empty( $vuln['sources'] ) ) : ?>
									<div class="oopvulns-sources">
										<strong><?php esc_html_e( 'References:', 'oopvulns-vulnerability-scanner' ); ?></strong>
										<ul>
											<?php foreach ( $vuln['sources'] as $source ) : ?>
												<li>
													<a href="<?php echo esc_url( $source['link'] ); ?>" target="_blank" rel="noopener noreferrer">
														<?php echo esc_html( $source['name'] ); ?>
														<span class="screen-reader-text"><?php esc_html_e( '(opens in new tab)', 'oopvulns-vulnerability-scanner' ); ?></span>
													</a>
												</li>
											<?php endforeach; ?>
										</ul>
									</div>
								<?php endif; ?>
							</div>
						<?php endforeach; ?>

						<div class="oopvulns-action">
							<?php
							// Check if update is available
							$update_plugins = get_site_transient( 'update_plugins' );
							$has_update = ! empty( $update_plugins->response[ $plugin['file'] ] );
							?>
							<?php if ( $has_update ) : ?>
								<a href="<?php echo esc_url( admin_url( 'plugins.php' ) ); ?>" class="oopvulns-action-btn" style="background: #000000; color: #FCDD56;">
									<span class="dashicons dashicons-update" aria-hidden="true" style="color: #FCDD56;"></span>
									<?php esc_html_e( 'Update Available - Update Now', 'oopvulns-vulnerability-scanner' ); ?>
								</a>
							<?php else : ?>
								<a href="<?php echo esc_url( admin_url( 'plugins.php' ) ); ?>" class="oopvulns-action-btn">
									<?php esc_html_e( 'Manage Plugins', 'oopvulns-vulnerability-scanner' ); ?>
								</a>
							<?php endif; ?>
						</div>
					</div>
				</div>
			<?php endforeach; ?>
		</div>
		<?php
	}

	/**
	 * Render theme vulnerabilities section.
	 *
	 * @param array $themes_result Themes scan result.
	 */
	private function render_theme_vulnerabilities( $themes_result ) {
		if ( empty( $themes_result['vulnerable'] ) ) {
			return;
		}
		?>
		<div class="oopvulns-section">
			<h2 class="oopvulns-section-title">
				<span class="dashicons dashicons-admin-appearance" aria-hidden="true"></span>
				<?php esc_html_e( 'Theme Vulnerabilities', 'oopvulns-vulnerability-scanner' ); ?>
				<span class="oopvulns-section-badge"><?php echo absint( $themes_result['vulnerable'] ); ?></span>
			</h2>

			<?php foreach ( $themes_result['themes'] as $theme ) : ?>
				<?php if ( ! $theme['vulnerable'] ) continue; ?>
				<?php $card_id = 'theme-' . sanitize_html_class( $theme['slug'] ); ?>
				<?php
				$deduped_theme_vulns      = $this->deduplicate_vulnerabilities( $theme['vulnerabilities'] ?? array() );
				$theme_vulnerabilities    = $deduped_theme_vulns['items'];
				$theme_duplicates         = $deduped_theme_vulns['duplicates'];
				$theme_duplicates_count   = $deduped_theme_vulns['count'];
				?>

				<div class="oopvulns-card oopvulns-card-warning oopvulns-collapsible" data-card-id="<?php echo esc_attr( $card_id ); ?>">
					<button type="button" class="oopvulns-card-header" aria-expanded="false" aria-controls="<?php echo esc_attr( $card_id ); ?>-body">
						<h3>
							<?php echo esc_html( $theme['name'] ); ?>
							<span class="oopvulns-version-badge">v<?php echo esc_html( $theme['version'] ); ?></span>
							<?php if ( $theme['active'] ) : ?>
								<span class="oopvulns-badge oopvulns-badge-info"><?php esc_html_e( 'Active', 'oopvulns-vulnerability-scanner' ); ?></span>
							<?php endif; ?>
						</h3>
						<span class="oopvulns-card-toggle">
							<span class="dashicons dashicons-arrow-down-alt2" aria-hidden="true"></span>
						</span>
					</button>

					<div class="oopvulns-card-body" id="<?php echo esc_attr( $card_id ); ?>-body" hidden>
						<?php if ( $theme_duplicates_count > 0 ) : ?>
							<div class="oopvulns-duplicates-notice">
								<p class="description">
									<span class="oopvulns-duplicates-count">
										<?php
										printf(
											/* translators: %d: number of duplicate vulnerabilities hidden */
											esc_html__( '%d possible duplicate vulnerability entries were hidden.', 'oopvulns-vulnerability-scanner' ),
											absint( $theme_duplicates_count )
										);
										?>
									</span>
									<button type="button" class="oopvulns-toggle-duplicates" data-section="<?php echo esc_attr( $card_id ); ?>">
										<span class="oopvulns-toggle-show"><?php esc_html_e( 'Show', 'oopvulns-vulnerability-scanner' ); ?></span>
										<span class="oopvulns-toggle-hide" style="display: none;"><?php esc_html_e( 'Hide', 'oopvulns-vulnerability-scanner' ); ?></span>
									</button>
								</p>
								<div class="oopvulns-hidden-duplicates" id="<?php echo esc_attr( $card_id ); ?>-duplicates" style="display: none;">
									<?php foreach ( $theme_duplicates as $dup_vuln ) : ?>
										<div class="oopvulns-vulnerability oopvulns-duplicate-highlight">
											<div class="oopvulns-vuln-header">
												<h4 class="oopvulns-vuln-title"><?php echo esc_html( $dup_vuln['name'] ); ?></h4>
												<?php $this->render_severity_badge( $dup_vuln['severity'], $dup_vuln['cvss_score'] ); ?>
											</div>
											<p class="description" style="margin-top: 8px; font-style: italic;"><?php esc_html_e( '(Duplicate)', 'oopvulns-vulnerability-scanner' ); ?></p>
										</div>
									<?php endforeach; ?>
								</div>
							</div>
						<?php endif; ?>
						<?php foreach ( $theme_vulnerabilities as $vuln ) : ?>
							<div class="oopvulns-vulnerability">
								<div class="oopvulns-vuln-header">
									<h4 class="oopvulns-vuln-title"><?php echo esc_html( $vuln['name'] ); ?></h4>
									<?php $this->render_severity_badge( $vuln['severity'], $vuln['cvss_score'] ); ?>
								</div>

								<?php if ( ! empty( $vuln['description'] ) ) : ?>
									<p class="oopvulns-vuln-description"><?php echo wp_kses_post( $vuln['description'] ); ?></p>
								<?php endif; ?>

								<p class="oopvulns-affected-versions">
									<strong><?php esc_html_e( 'Affected:', 'oopvulns-vulnerability-scanner' ); ?></strong>
									<?php echo esc_html( $this->format_versions( $vuln['versions'] ) ); ?>
								</p>

								<?php if ( ! empty( $vuln['sources'] ) ) : ?>
									<div class="oopvulns-sources">
										<strong><?php esc_html_e( 'References:', 'oopvulns-vulnerability-scanner' ); ?></strong>
										<ul>
											<?php foreach ( $vuln['sources'] as $source ) : ?>
												<li>
													<a href="<?php echo esc_url( $source['link'] ); ?>" target="_blank" rel="noopener noreferrer">
														<?php echo esc_html( $source['name'] ); ?>
														<span class="screen-reader-text"><?php esc_html_e( '(opens in new tab)', 'oopvulns-vulnerability-scanner' ); ?></span>
													</a>
												</li>
											<?php endforeach; ?>
										</ul>
									</div>
								<?php endif; ?>
							</div>
						<?php endforeach; ?>

						<div class="oopvulns-action">
							<a href="<?php echo esc_url( admin_url( 'themes.php' ) ); ?>" class="oopvulns-action-btn">
								<?php esc_html_e( 'Manage Themes', 'oopvulns-vulnerability-scanner' ); ?>
							</a>
						</div>
					</div>
				</div>
			<?php endforeach; ?>
		</div>
		<?php
	}

	/**
	 * Render severity badge.
	 *
	 * @param string $severity Severity level.
	 * @param float  $score CVSS score.
	 */
	private function render_severity_badge( $severity, $score ) {
		$severity_class = 'oopvulns-severity-' . strtolower( $severity );
		?>
		<span class="oopvulns-severity-badge <?php echo esc_attr( $severity_class ); ?>"
		      aria-label="<?php
				/* translators: %s: Severity level (Critical, High, Medium, Low) */
				echo esc_attr( sprintf( __( 'Severity: %s', 'oopvulns-vulnerability-scanner' ), ucfirst( $severity ) ) ); ?>">
			<span class="oopvulns-severity-level"><?php echo esc_html( ucfirst( $severity ) ); ?></span>
			<span class="oopvulns-severity-score"><?php echo esc_html( number_format( $score, 1 ) ); ?></span>
		</span>
		<?php
	}

	/**
	 * Render settings tab.
	 *
	 * @param array $settings Plugin settings.
	 */
	private function render_settings_tab( $settings ) {
		?>
		<div id="settings-panel" role="tabpanel" aria-labelledby="settings-tab" class="oopvulns-panel">
			<form method="post" action="options.php" class="oopvulns-form">
				<?php
				settings_fields( 'oopvulns_settings' );
				?>

				<div class="oopvulns-settings-section">
					<h2><?php esc_html_e( 'API Settings', 'oopvulns-vulnerability-scanner' ); ?></h2>
					<p class="description">
						<?php esc_html_e( 'Configure your API key to authenticate with the vulnerability scanner backend.', 'oopvulns-vulnerability-scanner' ); ?>
					</p>

					<div class="notice notice-warning inline" style="margin: 12px 0;">
						<p>
							<strong><?php esc_html_e( 'External Service:', 'oopvulns-vulnerability-scanner' ); ?></strong>
							<?php esc_html_e( 'This plugin connects to the OOPSpam API to check for vulnerabilities. Plugin/theme slugs and version numbers are sent to retrieve vulnerability data. No personal information is transmitted.', 'oopvulns-vulnerability-scanner' ); ?>
							<a href="https://www.oopspam.com/privacypolicy" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Privacy Policy', 'oopvulns-vulnerability-scanner' ); ?></a>
						</p>
					</div>

				<?php
				$api_key_info = $this->get_api_key_info();
				$api_key      = $api_key_info['key'];
				$source       = $api_key_info['source'];
				?>

				<?php if ( 'constant' === $source && ! empty( $api_key ) ) : ?>
					<div class="notice notice-info inline" style="margin: 12px 0;">
						<p>
							<strong><?php esc_html_e( 'API Key Source:', 'oopvulns-vulnerability-scanner' ); ?></strong>
							<?php esc_html_e( 'Using OOPSPAM_API_KEY constant from wp-config.php', 'oopvulns-vulnerability-scanner' ); ?>
						</p>
					</div>
				<?php elseif ( 'oopspam' === $source && ! empty( $api_key ) ) : ?>
					<div class="notice notice-info inline" style="margin: 12px 0;">
						<p>
							<strong><?php esc_html_e( 'API Key Source:', 'oopvulns-vulnerability-scanner' ); ?></strong>
							<?php esc_html_e( 'Using API key from OOPSpam Anti-Spam plugin settings', 'oopvulns-vulnerability-scanner' ); ?>
						</p>
					</div>
				<?php endif; ?>

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">
							<label for="api_key"><?php esc_html_e( 'API Key', 'oopvulns-vulnerability-scanner' ); ?></label>
						</th>
						<td>
							<?php if ( ! empty( $api_key ) && 'local' !== $source ) : ?>
								<input type="text" 
								       id="api_key" 
								       class="regular-text" 
								       disabled
								       value="<?php echo esc_attr( '(Configured via ' . ( 'constant' === $source ? 'wp-config.php' : 'OOP Spam plugin' ) . ')' ); ?>">
								<p class="description">
									<?php esc_html_e( 'This field is disabled because your API key is configured from an external source.', 'oopvulns-vulnerability-scanner' ); ?>
								</p>
							<?php else : ?>
								<input type="text" 
								       name="oopvulns_settings[api_key]" 
								       id="api_key" 
								       value="<?php echo esc_attr( $settings['api_key'] ?? '' ); ?>" 
								       class="regular-text"
								       placeholder="<?php esc_attr_e( 'Enter your API key', 'oopvulns-vulnerability-scanner' ); ?>"
								       aria-describedby="api-key-description">
								<p class="description" id="api-key-description">
									<?php esc_html_e( 'Required for vulnerability checks. Each API key includes 40 checks per month. Each plugin/theme check counts as one API call. You can also set this via the OOPSPAM_API_KEY constant in wp-config.php or use the OOPSpam Anti-Spam plugin API key.', 'oopvulns-vulnerability-scanner' ); ?>
								</p>
							<?php endif; ?>
						</td>
					</tr>
				</table>
			</div>

			<div class="oopvulns-settings-section">
				<h2><?php esc_html_e( 'Cache Settings', 'oopvulns-vulnerability-scanner' ); ?></h2>
				<p class="description">
					<?php esc_html_e( 'Configure how long vulnerability data should be cached to reduce API requests.', 'oopvulns-vulnerability-scanner' ); ?>
				</p>

					<table class="form-table" role="presentation">
						<tr>
							<th scope="row">
								<label for="cache_hours"><?php esc_html_e( 'Cache Duration', 'oopvulns-vulnerability-scanner' ); ?></label>
							</th>
							<td>
								<select name="oopvulns_settings[cache_hours]" id="cache_hours" class="oopvulns-select">
									<option value="1" <?php selected( $settings['cache_hours'] ?? 12, 1 ); ?>>
										<?php esc_html_e( '1 hour', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
									<option value="6" <?php selected( $settings['cache_hours'] ?? 12, 6 ); ?>>
										<?php esc_html_e( '6 hours', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
									<option value="12" <?php selected( $settings['cache_hours'] ?? 12, 12 ); ?>>
										<?php esc_html_e( '12 hours (recommended)', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
									<option value="24" <?php selected( $settings['cache_hours'] ?? 12, 24 ); ?>>
										<?php esc_html_e( '24 hours', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
								</select>
								<p class="description">
									<?php esc_html_e( 'Longer cache duration reduces API calls but may delay detection of new vulnerabilities.', 'oopvulns-vulnerability-scanner' ); ?>
								</p>
							</td>
						</tr>
					</table>
				</div>

				<div class="oopvulns-settings-section">
					<h2><?php esc_html_e( 'Scan Settings', 'oopvulns-vulnerability-scanner' ); ?></h2>
					<p class="description">
						<?php esc_html_e( 'Configure automatic vulnerability scanning schedule.', 'oopvulns-vulnerability-scanner' ); ?>
					</p>

					<table class="form-table" role="presentation">
						<tr>
							<th scope="row">
								<label for="scan_frequency"><?php esc_html_e( 'Scan Frequency', 'oopvulns-vulnerability-scanner' ); ?></label>
							</th>
							<td>
								<select name="oopvulns_settings[scan_frequency]" id="scan_frequency" class="oopvulns-select">
									<option value="daily" <?php selected( $settings['scan_frequency'] ?? 'daily', 'daily' ); ?>>
										<?php esc_html_e( 'Daily', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
									<option value="weekly" <?php selected( $settings['scan_frequency'] ?? 'daily', 'weekly' ); ?>>
										<?php esc_html_e( 'Weekly', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
								</select>
								<p class="description">
									<?php esc_html_e( 'How often automatic vulnerability scans should run. More frequent scans help detect new vulnerabilities faster.', 'oopvulns-vulnerability-scanner' ); ?>
								</p>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<?php esc_html_e( 'Next Scheduled Scan', 'oopvulns-vulnerability-scanner' ); ?>
							</th>
							<td>
								<?php
								$next_scan = wp_next_scheduled( 'oopvulns_scan' );
								?>
								<p class="description">
									<?php if ( $next_scan ) : ?>
										<?php
										printf(
											/* translators: 1: Date and time, 2: Relative time */
											esc_html__( '%1$s (%2$s from now)', 'oopvulns-vulnerability-scanner' ),
											esc_html( wp_date( 'M j, Y H:i', $next_scan ) ),
											esc_html( human_time_diff( current_time( 'timestamp' ), $next_scan ) )
										);
										?>
									<?php else : ?>
										<?php esc_html_e( 'Not scheduled yet. Save settings to schedule the next scan.', 'oopvulns-vulnerability-scanner' ); ?>
									<?php endif; ?>
								</p>
							</td>
						</tr>
					</table>
				</div>

				<div class="oopvulns-settings-section">
					<h2><?php esc_html_e( 'Email Notifications', 'oopvulns-vulnerability-scanner' ); ?></h2>
					<p class="description">
						<?php esc_html_e( 'Configure automatic email notifications for vulnerability alerts.', 'oopvulns-vulnerability-scanner' ); ?>
					</p>

					<table class="form-table" role="presentation">
						<tr>
							<th scope="row">
								<label for="enabled">
									<?php esc_html_e( 'Enable Notifications', 'oopvulns-vulnerability-scanner' ); ?>
								</label>
							</th>
							<td>
								<label for="enabled">
									<input type="checkbox" 
									       name="oopvulns_settings[enabled]" 
									       id="enabled" 
									       value="1" 
									       <?php checked( ! empty( $settings['enabled'] ) ); ?>>
									<?php esc_html_e( 'Send email notifications when vulnerabilities are detected', 'oopvulns-vulnerability-scanner' ); ?>
								</label>
							</td>
						</tr>

						<tr>
							<th scope="row">
								<label for="notification_email"><?php esc_html_e( 'Email Address', 'oopvulns-vulnerability-scanner' ); ?></label>
							</th>
							<td>
								<input type="email" 
								       name="oopvulns_settings[notification_email]" 
								       id="notification_email" 
								       value="<?php echo esc_attr( $settings['notification_email'] ?? get_option( 'admin_email' ) ); ?>" 
								       class="regular-text"
								       aria-describedby="notification-email-description">
								<p class="description" id="notification-email-description">
									<?php esc_html_e( 'Email address to receive vulnerability notifications. Defaults to the Administration Email Address.', 'oopvulns-vulnerability-scanner' ); ?>
								</p>
							</td>
						</tr>

						<tr>
							<th scope="row">
								<label for="notification_schedule"><?php esc_html_e( 'Notification Frequency', 'oopvulns-vulnerability-scanner' ); ?></label>
							</th>
							<td>
								<select name="oopvulns_settings[notification_schedule]" id="notification_schedule" class="oopvulns-select">
									<option value="daily" <?php selected( $settings['notification_schedule'] ?? 'weekly', 'daily' ); ?>>
										<?php esc_html_e( 'Daily', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
									<option value="weekly" <?php selected( $settings['notification_schedule'] ?? 'weekly', 'weekly' ); ?>>
										<?php esc_html_e( 'Weekly', 'oopvulns-vulnerability-scanner' ); ?>
									</option>
								</select>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<?php esc_html_e( 'Next Scheduled Notification', 'oopvulns-vulnerability-scanner' ); ?>
							</th>
							<td>
								<?php
								$next_notification = wp_next_scheduled( 'oopvulns_notify' );
								?>
								<p class="description">
									<?php if ( $next_notification ) : ?>
										<?php
										printf(
											/* translators: 1: Date and time, 2: Relative time */
											esc_html__( '%1$s (%2$s from now)', 'oopvulns-vulnerability-scanner' ),
											esc_html( wp_date( 'M j, Y H:i', $next_notification ) ),
											esc_html( human_time_diff( current_time( 'timestamp' ), $next_notification ) )
										);
										?>
									<?php else : ?>
										<?php esc_html_e( 'Not scheduled yet. Save settings to schedule the next notification.', 'oopvulns-vulnerability-scanner' ); ?>
									<?php endif; ?>
								</p>
							</td>
						</tr>

						<tr id="notification-day-row" style="<?php echo ( isset( $settings['notification_schedule'] ) && $settings['notification_schedule'] === 'daily' ) ? 'display:none;' : ''; ?>">
							<th scope="row">
								<label for="notification_day"><?php esc_html_e( 'Day of Week', 'oopvulns-vulnerability-scanner' ); ?></label>
							</th>
							<td>
								<select name="oopvulns_settings[notification_day]" id="notification_day" class="oopvulns-select">
									<?php
									$days = array(
										'monday'    => __( 'Monday', 'oopvulns-vulnerability-scanner' ),
										'tuesday'   => __( 'Tuesday', 'oopvulns-vulnerability-scanner' ),
										'wednesday' => __( 'Wednesday', 'oopvulns-vulnerability-scanner' ),
										'thursday'  => __( 'Thursday', 'oopvulns-vulnerability-scanner' ),
										'friday'    => __( 'Friday', 'oopvulns-vulnerability-scanner' ),
										'saturday'  => __( 'Saturday', 'oopvulns-vulnerability-scanner' ),
										'sunday'    => __( 'Sunday', 'oopvulns-vulnerability-scanner' ),
									);
									foreach ( $days as $value => $label ) :
										?>
										<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $settings['notification_day'] ?? 'monday', $value ); ?>>
											<?php echo esc_html( $label ); ?>
										</option>
									<?php endforeach; ?>
								</select>
							</td>
						</tr>

						<tr>
							<th scope="row">
								<label for="notification_hour"><?php esc_html_e( 'Time', 'oopvulns-vulnerability-scanner' ); ?></label>
							</th>
							<td>
								<select name="oopvulns_settings[notification_hour]" id="notification_hour" class="oopvulns-select-inline">
									<?php for ( $h = 0; $h < 24; $h++ ) : ?>
										<option value="<?php echo absint( $h ); ?>" <?php selected( $settings['notification_hour'] ?? 9, $h ); ?>>
											<?php echo esc_html( sprintf( '%02d', $h ) ); ?>
										</option>
									<?php endfor; ?>
								</select>
								<span>:</span>
								<select name="oopvulns_settings[notification_minute]" id="notification_minute" class="oopvulns-select-inline">
									<?php for ( $m = 0; $m < 60; $m += 15 ) : ?>
										<option value="<?php echo absint( $m ); ?>" <?php selected( $settings['notification_minute'] ?? 0, $m ); ?>>
											<?php echo esc_html( sprintf( '%02d', $m ) ); ?>
										</option>
									<?php endfor; ?>
								</select>
								<p class="description">
									<?php esc_html_e( 'Time to send notifications (in your local timezone).', 'oopvulns-vulnerability-scanner' ); ?>
								</p>
							</td>
						</tr>
					</table>
				</div>

				<?php submit_button( __( 'Save Settings', 'oopvulns-vulnerability-scanner' ), 'primary', 'submit', true, array( 'id' => 'submit-settings' ) ); ?>
			</form>

			<div class="oopvulns-danger-zone">
				<h2><?php esc_html_e( 'Actions', 'oopvulns-vulnerability-scanner' ); ?></h2>
				
				<div class="oopvulns-action-group">
					<form method="post" action="" style="display: inline;">
						<?php wp_nonce_field( 'oopvulns_action' ); ?>
						<input type="hidden" name="oopvulns_action" value="send_test_email">
						<button type="submit" class="button">
							<span class="dashicons dashicons-email" aria-hidden="true"></span>
							<?php esc_html_e( 'Send Test Email', 'oopvulns-vulnerability-scanner' ); ?>
						</button>
					</form>

					<form method="post" action="" style="display: inline;">
						<?php wp_nonce_field( 'oopvulns_action' ); ?>
						<input type="hidden" name="oopvulns_action" value="clear_cache">
						<button type="submit" class="button" onclick="return confirm('<?php echo esc_js( __( 'Are you sure you want to clear all cached vulnerability data?', 'oopvulns-vulnerability-scanner' ) ); ?>');">
							<span class="dashicons dashicons-trash" aria-hidden="true"></span>
							<?php esc_html_e( 'Clear Cache', 'oopvulns-vulnerability-scanner' ); ?>
						</button>
					</form>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Enqueue styles for plugins.php and themes.php vulnerability notices.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_vulnerability_page_styles( $hook ) {
		if ( ! in_array( $hook, array( 'tools_page_oopvulns-vulnerability-scanner', 'plugins.php', 'themes.php' ), true ) ) {
			return;
		}

		wp_add_inline_style( 'wp-admin', '
			/* Modern inline vulnerability notice row */
			tr.oopvulns-vuln-row td {
				padding: 0 !important;
				background: transparent !important;
				border: none !important;
				box-shadow: none !important;
			}
			.oopvulns-inline-vuln-card {
				background: linear-gradient(135deg, #fffbeb 0%, #fef9e7 100%);
				border-left: 3px solid #000000;
				border-radius: 6px;
				padding: 14px 18px;
				margin: 8px 0;
				display: flex;
				align-items: center;
				gap: 16px;
				transition: all 0.2s ease;
			}
			.oopvulns-inline-vuln-card:hover {
				box-shadow: 0 2px 8px rgba(0,0,0,0.08);
				transform: translateX(2px);
			}
			.oopvulns-inline-vuln-icon {
				flex-shrink: 0;
				width: 32px;
				height: 32px;
				display: flex;
				align-items: center;
				justify-content: center;
				background: #000000;
				color: #FCDD56;
				border-radius: 50%;
				font-size: 16px;
				font-weight: 700;
			}
			.oopvulns-inline-vuln-content {
				flex: 1;
				min-width: 0;
			}
			.oopvulns-inline-vuln-header {
				display: flex;
				align-items: center;
				gap: 10px;
				margin-bottom: 6px;
				flex-wrap: wrap;
			}
			.oopvulns-inline-vuln-count {
				font-size: 13px;
				font-weight: 700;
				color: #000000;
				background: rgba(0,0,0,0.08);
				padding: 3px 10px;
				border-radius: 12px;
			}
			.oopvulns-inline-vuln-meta {
				display: flex;
				align-items: center;
				gap: 12px;
				flex-wrap: wrap;
				font-size: 12px;
				color: #000000;
			}
			.oopvulns-inline-vuln-versions {
				display: flex;
				align-items: center;
				gap: 4px;
				font-weight: 500;
				color: #000000;
			}
			.oopvulns-inline-vuln-versions svg {
				width: 14px;
				height: 14px;
				opacity: 0.6;
			}
			.oopvulns-inline-severity {
				display: inline-flex;
				align-items: center;
				padding: 3px 10px;
				border-radius: 12px;
				font-size: 11px;
				font-weight: 700;
				text-transform: uppercase;
				letter-spacing: 0.3px;
			}
			/* Severity badges - minimal grayscale */
			.oopvulns-severity-critical {
				background: #1a1a1a;
				color: #ffffff;
			}
			.oopvulns-severity-high {
				background: #333333;
				color: #ffffff;
			}
			.oopvulns-severity-medium {
				background: #e0e0e0;
				color: #1a1a1a;
			}
			.oopvulns-severity-low {
				background: #f5f5f5;
				color: #1a1a1a;
			}
			.oopvulns-inline-status {
				display: inline-flex;
				align-items: center;
				gap: 4px;
				font-weight: 600;
			}
			.oopvulns-inline-status.has-update {
				color: #000000;
				background: #FCDD56;
				padding: 2px 8px;
				border-radius: 10px;
			}
			.oopvulns-inline-status.no-fix {
				color: #000000;
				font-weight: 700;
			}
			.oopvulns-inline-status.closed {
				color: #000000;
				font-weight: 700;
			}
			.oopvulns-inline-action {
				flex-shrink: 0;
			}
			.oopvulns-inline-action a {
				display: inline-flex;
				align-items: center;
				gap: 6px;
				padding: 6px 14px;
				background: #000000;
				color: #FCDD56;
				border-radius: 4px;
				font-size: 12px;
				font-weight: 600;
				text-decoration: none;
				transition: all 0.2s ease;
			}
			.oopvulns-inline-action a:hover {
				background: #1a1a1a;
				color: #FCDD56;
				transform: translateY(-1px);
				box-shadow: 0 2px 4px rgba(0,0,0,0.15);
			}
			.oopvulns-inline-action a.has-update {
				background: #FCDD56;
				color: #000000;
			}
			.oopvulns-inline-action a.has-update:hover {
				background: #e5c338;
			}
			/* Vertical vulnerability list */
			.oopvulns-inline-vuln-card-expanded {
				flex-direction: column;
				align-items: stretch;
				gap: 12px;
			}
			.oopvulns-inline-vuln-card-expanded .oopvulns-inline-vuln-icon {
				position: absolute;
				top: 14px;
				left: 18px;
			}
			.oopvulns-inline-vuln-card-expanded {
				position: relative;
				padding-left: 60px;
			}
			.oopvulns-inline-vuln-card-expanded .oopvulns-inline-action {
				align-self: flex-start;
			}
			.oopvulns-inline-vuln-list {
				list-style: none;
				margin: 12px 0 0 0;
				padding: 0;
				display: flex;
				flex-direction: column;
				gap: 8px;
			}
			.oopvulns-inline-vuln-list-item {
				display: flex;
				align-items: center;
				gap: 10px;
				padding: 10px 14px;
				background: rgba(255, 255, 255, 0.7);
				border-radius: 6px;
				border: 1px solid rgba(0, 0, 0, 0.06);
				flex-wrap: wrap;
			}
			.oopvulns-inline-vuln-list-item:hover {
				background: rgba(255, 255, 255, 0.9);
				border-color: rgba(0, 0, 0, 0.1);
			}
			.oopvulns-inline-vuln-name {
				font-weight: 600;
				color: #000000;
				font-size: 13px;
				flex: 1;
				min-width: 200px;
			}
			.oopvulns-inline-vuln-versions-detail {
				font-size: 12px;
				color: #000000;
				opacity: 0.7;
				padding: 4px 10px;
				background: rgba(0, 0, 0, 0.05);
				border-radius: 10px;
				white-space: nowrap;
			}
			/* Show more/less button */
			.oopvulns-show-more-vulns {
				background: transparent;
				border: none;
				color: #000000;
				font-size: 12px;
				font-weight: 600;
				padding: 8px 12px;
				cursor: pointer;
				border-radius: 4px;
				margin-top: 8px;
				transition: all 0.2s ease;
			}
			.oopvulns-show-more-vulns:hover {
				background: rgba(0, 0, 0, 0.05);
			}
			.oopvulns-show-more-vulns:focus {
				outline: 2px solid #FCDD56;
				outline-offset: 2px;
			}
			/* Theme overlay vulnerability notice */
			.oopvulns-theme-overlay-notice {
				background: linear-gradient(135deg, #fffbeb 0%, #fef9e7 100%);
				border-left: 3px solid #000000;
				border-radius: 6px;
				padding: 12px 16px;
				margin: 10px 0;
				font-size: 13px;
			}
			.oopvulns-theme-overlay-notice strong {
				color: #000000;
			}
			/* Duplicates notice and toggle */
			.oopvulns-duplicates-notice {
				background: rgba(0, 0, 0, 0.04);
				border-left: 2px solid #FCDD56;
				padding: 10px 12px;
				border-radius: 4px;
				margin: 12px 0;
			}
			.oopvulns-duplicates-notice .description {
				margin: 0;
				display: flex;
				align-items: center;
				gap: 8px;
				flex-wrap: wrap;
			}
			.oopvulns-toggle-duplicates {
				background: transparent;
				border: 1px solid #FCDD56;
				color: #000000;
				padding: 4px 12px;
				border-radius: 4px;
				font-size: 12px;
				font-weight: 600;
				cursor: pointer;
				transition: all 0.2s ease;
			}
			.oopvulns-toggle-duplicates:hover {
				background: #FCDD56;
				color: #000000;
			}
			.oopvulns-toggle-duplicates:focus {
				outline: 2px solid #FCDD56;
				outline-offset: 2px;
			}
			.oopvulns-duplicate-highlight {
				opacity: 0.9;
				border-left: 3px solid #FCDD56;
				background: rgba(252, 221, 86, 0.08);
			}
			.oopvulns-hidden-duplicates {
				margin-top: 12px;
				padding-top: 12px;
				border-top: 1px solid rgba(0, 0, 0, 0.1);
			}
			.oopvulns-hidden-duplicates-inline {
				background: rgba(252, 221, 86, 0.08);
				padding: 8px;
				border-radius: 4px;
			}
		' );

		// Add JavaScript for show more/less functionality and toggle duplicates
		wp_add_inline_script( 'jquery', '
			jQuery(document).ready(function($) {
				$(document).on("click", ".oopvulns-show-more-vulns", function(e) {
					e.preventDefault();
					var $btn = $(this);
					var $card = $btn.closest(".oopvulns-inline-vuln-card");
					var $hiddenVulns = $card.find(".oopvulns-hidden-vuln");
					var $showMoreText = $btn.find(".oopvulns-show-more-text");
					var $showLessText = $btn.find(".oopvulns-show-less-text");
					
					if ($hiddenVulns.first().is(":visible")) {
						// Hide them
						$hiddenVulns.slideUp(200);
						$showMoreText.show();
						$showLessText.hide();
					} else {
						// Show them
						$hiddenVulns.slideDown(200);
						$showMoreText.hide();
						$showLessText.show();
					}
				});
				
				// Toggle hidden duplicates
				$(document).on("click", ".oopvulns-toggle-duplicates", function(e) {
					e.preventDefault();
					var $btn = $(this);
					var section = $btn.data("section");
					var $duplicates = $("#" + section + "-duplicates");
					var $showText = $btn.find(".oopvulns-toggle-show");
					var $hideText = $btn.find(".oopvulns-toggle-hide");
					
					if ($duplicates.is(":visible")) {
						$duplicates.slideUp(200);
						$showText.show();
						$hideText.hide();
					} else {
						$duplicates.slideDown(200);
						$showText.hide();
						$hideText.show();
					}
				});
			});
		' );
	}

	/**
	 * Register vulnerability notices below each vulnerable plugin on plugins.php.
	 */
	public function register_plugin_vulnerability_notices() {
		global $pagenow;
		if ( 'plugins.php' !== $pagenow || ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$plugins_result = get_option( 'oopvulns_plugins_result', array() );
		if ( empty( $plugins_result['plugins'] ) ) {
			return;
		}

		foreach ( $plugins_result['plugins'] as $plugin_data ) {
			if ( ! empty( $plugin_data['vulnerable'] ) && ! empty( $plugin_data['file'] ) ) {
				$file = $plugin_data['file'];
				add_action(
					'after_plugin_row_' . $file,
					function ( $plugin_file, $wp_plugin_data ) use ( $plugin_data ) {
						$this->render_plugin_vulnerability_row( $plugin_file, $wp_plugin_data, $plugin_data );
					},
					10,
					2
				);
			}
		}
	}

	/**
	 * Render a vulnerability notice row below a plugin on plugins.php.
	 *
	 * @param string $plugin_file Plugin file path.
	 * @param array  $wp_plugin_data WordPress plugin data.
	 * @param array  $scan_data Our vulnerability scan data for this plugin.
	 */
	private function render_plugin_vulnerability_row( $plugin_file, $wp_plugin_data, $scan_data ) {
		$tr_class = is_plugin_active( $plugin_file ) ? 'active' : '';
		$columns  = count( get_column_headers( get_current_screen() ) );
		$deduped_plugin_vulns     = $this->deduplicate_vulnerabilities( $scan_data['vulnerabilities'] ?? array() );
		$plugin_vulnerabilities   = $deduped_plugin_vulns['items'];
		$plugin_duplicates        = $deduped_plugin_vulns['duplicates'];
		$plugin_duplicates_count  = $deduped_plugin_vulns['count'];
		$vuln_count               = count( $plugin_vulnerabilities );

		// Check for updates
		$update_plugins   = get_site_transient( 'update_plugins' );
		$update_available = ! empty( $update_plugins->response[ $plugin_file ] );

		// Get highest severity
		$highest_severity = 'low';
		$has_unfixed = false;

		foreach ( $plugin_vulnerabilities as $vuln ) {
			if ( ! empty( $vuln['unfixed'] ) ) {
				$has_unfixed = true;
			}
			$severity = strtolower( $vuln['severity'] ?? 'low' );
			$severity_levels = array( 'low' => 1, 'medium' => 2, 'high' => 3, 'critical' => 4 );
			if ( isset( $severity_levels[ $severity ] ) && $severity_levels[ $severity ] > $severity_levels[ $highest_severity ] ) {
				$highest_severity = $severity;
			}
		}
		?>
		<tr class="oopvulns-vuln-row plugin-update-tr <?php echo esc_attr( $tr_class ); ?>">
			<td colspan="<?php echo absint( $columns ); ?>">
				<div class="oopvulns-inline-vuln-card oopvulns-inline-vuln-card-expanded">
					<div class="oopvulns-inline-vuln-icon">⚠</div>
					
					<div class="oopvulns-inline-vuln-content">
						<div class="oopvulns-inline-vuln-header">
							<span class="oopvulns-inline-vuln-count">
								<?php
								printf(
									/* translators: %d: number of vulnerabilities */
									esc_html( _n( '%d Vulnerability Found', '%d Vulnerabilities Found', $vuln_count, 'oopvulns-vulnerability-scanner' ) ),
									absint( $vuln_count )
								);
								?>
							</span>
							<?php if ( $update_available ) : ?>
								<span class="oopvulns-inline-status has-update">✓ Update Available</span>
							<?php elseif ( $has_unfixed ) : ?>
								<span class="oopvulns-inline-status no-fix">Update to Latest &amp; Re-scan</span>
							<?php endif; ?>
						</div>
						<?php if ( $plugin_duplicates_count > 0 ) : ?>
							<div class="oopvulns-duplicates-notice">
								<p class="description">
									<span class="oopvulns-duplicates-count">
										<?php
										printf(
											/* translators: %d: number of duplicate vulnerabilities hidden */
											esc_html__( '%d possible duplicate vulnerability entries were hidden.', 'oopvulns-vulnerability-scanner' ),
											absint( $plugin_duplicates_count )
										);
										?>
									</span>
									<button type="button" class="oopvulns-toggle-duplicates" data-section="plugin-<?php echo esc_attr( sanitize_html_class( $plugin_file ) ); ?>">
										<span class="oopvulns-toggle-show"><?php esc_html_e( 'Show', 'oopvulns-vulnerability-scanner' ); ?></span>
										<span class="oopvulns-toggle-hide" style="display: none;"><?php esc_html_e( 'Hide', 'oopvulns-vulnerability-scanner' ); ?></span>
									</button>
								</p>
								<ul class="oopvulns-inline-vuln-list oopvulns-hidden-duplicates-inline" id="plugin-<?php echo esc_attr( sanitize_html_class( $plugin_file ) ); ?>-duplicates" style="display: none; margin-top: 12px;">
									<?php foreach ( $plugin_duplicates as $dup_vuln ) : ?>
										<li class="oopvulns-inline-vuln-list-item oopvulns-duplicate-highlight">
											<span class="oopvulns-inline-severity oopvulns-severity-<?php echo esc_attr( strtolower( $dup_vuln['severity'] ?? 'low' ) ); ?>">
												<?php echo esc_html( ucfirst( $dup_vuln['severity'] ?? 'Unknown' ) ); ?>
											</span>
											<span class="oopvulns-inline-vuln-name"><?php echo esc_html( $dup_vuln['name'] ?? 'Security Vulnerability' ); ?></span>
											<span class="description" style="margin-left: 8px;"><?php esc_html_e( '(Duplicate)', 'oopvulns-vulnerability-scanner' ); ?></span>
										</li>
									<?php endforeach; ?>
								</ul>
							</div>
						<?php endif; ?>
						
						<ul class="oopvulns-inline-vuln-list">
							<?php 
							$max_visible = 3;
							$total_vulns = count( $plugin_vulnerabilities );
							foreach ( $plugin_vulnerabilities as $index => $vuln ) : 
								$hidden_class = ( $index >= $max_visible ) ? 'oopvulns-hidden-vuln' : '';
							?>
								<li class="oopvulns-inline-vuln-list-item <?php echo esc_attr( $hidden_class ); ?>" <?php echo ( $index >= $max_visible ) ? 'style="display: none;"' : ''; ?>>
									<span class="oopvulns-inline-severity oopvulns-severity-<?php echo esc_attr( strtolower( $vuln['severity'] ?? 'low' ) ); ?>">
										<?php echo esc_html( ucfirst( $vuln['severity'] ?? 'Unknown' ) ); ?>
									</span>
									<span class="oopvulns-inline-vuln-name"><?php echo esc_html( $vuln['name'] ?? 'Security Vulnerability' ); ?></span>
									<span class="oopvulns-inline-vuln-versions-detail">
										<?php echo esc_html( $this->format_versions( $vuln['versions'] ?? '' ) ?: 'Affected versions unknown' ); ?>
									</span>
								</li>
							<?php endforeach; ?>
						</ul>
						<?php if ( $total_vulns > $max_visible ) : ?>
							<button type="button" class="oopvulns-show-more-vulns" data-plugin="<?php echo esc_attr( $plugin_file ); ?>">
								<span class="oopvulns-show-more-text">
									<?php 
									printf(
										/* translators: %d: number of hidden vulnerabilities */
										esc_html__( 'Show %d more', 'oopvulns-vulnerability-scanner' ),
										absint( $total_vulns - $max_visible )
									);
									?>
								</span>
								<span class="oopvulns-show-less-text" style="display: none;"><?php esc_html_e( 'Show less', 'oopvulns-vulnerability-scanner' ); ?></span>
							</button>
						<?php endif; ?>
					</div>

					<div class="oopvulns-inline-action">
						<?php if ( $update_available ) : ?>
							<a href="<?php echo esc_url( admin_url( 'update-core.php' ) ); ?>" class="has-update">
								<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 16 16" fill="currentColor" style="width:14px;height:14px">
									<path d="M8.75 2.75a.75.75 0 0 0-1.5 0v5.69L5.03 6.22a.75.75 0 0 0-1.06 1.06l3.5 3.5a.75.75 0 0 0 1.06 0l3.5-3.5a.75.75 0 0 0-1.06-1.06L8.75 8.44V2.75z"/>
									<path d="M3.5 9.75a.75.75 0 0 0-1.5 0v1.5A2.75 2.75 0 0 0 4.75 14h6.5A2.75 2.75 0 0 0 14 11.25v-1.5a.75.75 0 0 0-1.5 0v1.5c0 .69-.56 1.25-1.25 1.25h-6.5c-.69 0-1.25-.56-1.25-1.25v-1.5z"/>
								</svg>
								Update Now
							</a>
						<?php else : ?>
							<a href="<?php echo esc_url( admin_url( 'tools.php?page=oopvulns-vulnerability-scanner' ) ); ?>">
								View Details
							</a>
						<?php endif; ?>
					</div>
				</div>
			</td>
		</tr>
		<?php
	}

	/**
	 * Register vulnerability notices on themes.php.
	 */
	public function register_theme_vulnerability_notices() {
		global $pagenow;
		if ( 'themes.php' !== $pagenow || ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$themes_result = get_option( 'oopvulns_themes_result', array() );
		if ( empty( $themes_result['themes'] ) ) {
			return;
		}

		foreach ( $themes_result['themes'] as $slug => $theme_data ) {
			if ( ! empty( $theme_data['vulnerable'] ) ) {
				add_action(
					'after_theme_row_' . $slug,
					function ( $theme_stylesheet, $theme_obj ) use ( $theme_data ) {
						$this->render_theme_vulnerability_row( $theme_stylesheet, $theme_obj, $theme_data );
					},
					10,
					2
				);
			}
		}
	}

	/**
	 * Render a vulnerability notice row below a theme on themes.php.
	 *
	 * @param string   $stylesheet  Theme stylesheet.
	 * @param \WP_Theme $theme_obj  WordPress theme object.
	 * @param array    $scan_data   Our vulnerability scan data for this theme.
	 */
	private function render_theme_vulnerability_row( $stylesheet, $theme_obj, $scan_data ) {
		$current_theme = wp_get_theme();
		$tr_class      = ( $stylesheet === $current_theme->get_stylesheet() ) ? 'active' : '';
		$deduped_theme_vulns    = $this->deduplicate_vulnerabilities( $scan_data['vulnerabilities'] ?? array() );
		$theme_vulnerabilities  = $deduped_theme_vulns['items'];
		$theme_duplicates       = $deduped_theme_vulns['duplicates'];
		$theme_duplicates_count = $deduped_theme_vulns['count'];
		$vuln_count             = count( $theme_vulnerabilities );

		// Check for updates
		$update_themes    = get_site_transient( 'update_themes' );
		$update_available = ! empty( $update_themes->response[ $stylesheet ] );

		// Get highest severity
		$highest_severity = 'low';
		$has_unfixed = false;

		foreach ( $theme_vulnerabilities as $vuln ) {
			if ( ! empty( $vuln['unfixed'] ) ) {
				$has_unfixed = true;
			}
			$severity = strtolower( $vuln['severity'] ?? 'low' );
			$severity_levels = array( 'low' => 1, 'medium' => 2, 'high' => 3, 'critical' => 4 );
			if ( isset( $severity_levels[ $severity ] ) && $severity_levels[ $severity ] > $severity_levels[ $highest_severity ] ) {
				$highest_severity = $severity;
			}
		}
		?>
		<tr class="oopvulns-vuln-row <?php echo esc_attr( $tr_class ); ?>">
			<td colspan="4">
				<div class="oopvulns-inline-vuln-card oopvulns-inline-vuln-card-expanded">
					<div class="oopvulns-inline-vuln-icon">⚠</div>
					
					<div class="oopvulns-inline-vuln-content">
						<div class="oopvulns-inline-vuln-header">
							<span class="oopvulns-inline-vuln-count">
								<?php
								printf(
									/* translators: %d: number of vulnerabilities */
									esc_html( _n( '%d Vulnerability Found', '%d Vulnerabilities Found', $vuln_count, 'oopvulns-vulnerability-scanner' ) ),
									absint( $vuln_count )
								);
								?>
							</span>
							<?php if ( $update_available ) : ?>
								<span class="oopvulns-inline-status has-update">✓ Update Available</span>
							<?php elseif ( $has_unfixed ) : ?>
								<span class="oopvulns-inline-status no-fix">Update to Latest &amp; Re-scan</span>
							<?php endif; ?>
						</div>
						<?php if ( $theme_duplicates_count > 0 ) : ?>
							<div class="oopvulns-duplicates-notice">
								<p class="description">
									<span class="oopvulns-duplicates-count">
										<?php
										printf(
											/* translators: %d: number of duplicate vulnerabilities hidden */
											esc_html__( '%d possible duplicate vulnerability entries were hidden.', 'oopvulns-vulnerability-scanner' ),
											absint( $theme_duplicates_count )
										);
										?>
									</span>
									<button type="button" class="oopvulns-toggle-duplicates" data-section="theme-<?php echo esc_attr( sanitize_html_class( $stylesheet ) ); ?>">
										<span class="oopvulns-toggle-show"><?php esc_html_e( 'Show', 'oopvulns-vulnerability-scanner' ); ?></span>
										<span class="oopvulns-toggle-hide" style="display: none;"><?php esc_html_e( 'Hide', 'oopvulns-vulnerability-scanner' ); ?></span>
									</button>
								</p>
								<ul class="oopvulns-inline-vuln-list oopvulns-hidden-duplicates-inline" id="theme-<?php echo esc_attr( sanitize_html_class( $stylesheet ) ); ?>-duplicates" style="display: none; margin-top: 12px;">
									<?php foreach ( $theme_duplicates as $dup_vuln ) : ?>
										<li class="oopvulns-inline-vuln-list-item oopvulns-duplicate-highlight">
											<span class="oopvulns-inline-severity oopvulns-severity-<?php echo esc_attr( strtolower( $dup_vuln['severity'] ?? 'low' ) ); ?>">
												<?php echo esc_html( ucfirst( $dup_vuln['severity'] ?? 'Unknown' ) ); ?>
											</span>
											<span class="oopvulns-inline-vuln-name"><?php echo esc_html( $dup_vuln['name'] ?? 'Security Vulnerability' ); ?></span>
											<span class="description" style="margin-left: 8px;"><?php esc_html_e( '(Duplicate)', 'oopvulns-vulnerability-scanner' ); ?></span>
										</li>
									<?php endforeach; ?>
								</ul>
							</div>
						<?php endif; ?>
						
						<ul class="oopvulns-inline-vuln-list">
							<?php 
							$max_visible = 3;
							$total_vulns = count( $theme_vulnerabilities );
							foreach ( $theme_vulnerabilities as $index => $vuln ) : 
								$hidden_class = ( $index >= $max_visible ) ? 'oopvulns-hidden-vuln' : '';
							?>
								<li class="oopvulns-inline-vuln-list-item <?php echo esc_attr( $hidden_class ); ?>" <?php echo ( $index >= $max_visible ) ? 'style="display: none;"' : ''; ?>>
									<span class="oopvulns-inline-severity oopvulns-severity-<?php echo esc_attr( strtolower( $vuln['severity'] ?? 'low' ) ); ?>">
										<?php echo esc_html( ucfirst( $vuln['severity'] ?? 'Unknown' ) ); ?>
									</span>
									<span class="oopvulns-inline-vuln-name"><?php echo esc_html( $vuln['name'] ?? 'Security Vulnerability' ); ?></span>
									<span class="oopvulns-inline-vuln-versions-detail">
										<?php echo esc_html( $this->format_versions( $vuln['versions'] ?? '' ) ?: 'Affected versions unknown' ); ?>
									</span>
								</li>
							<?php endforeach; ?>
						</ul>
						<?php if ( $total_vulns > $max_visible ) : ?>
							<button type="button" class="oopvulns-show-more-vulns" data-theme="<?php echo esc_attr( $stylesheet ); ?>">
								<span class="oopvulns-show-more-text">
									<?php 
									printf(
										/* translators: %d: number of hidden vulnerabilities */
										esc_html__( 'Show %d more', 'oopvulns-vulnerability-scanner' ),
										absint( $total_vulns - $max_visible )
									);
									?>
								</span>
								<span class="oopvulns-show-less-text" style="display: none;"><?php esc_html_e( 'Show less', 'oopvulns-vulnerability-scanner' ); ?></span>
							</button>
						<?php endif; ?>
					</div>

					<div class="oopvulns-inline-action">
						<?php if ( $update_available ) : ?>
							<a href="<?php echo esc_url( admin_url( 'update-core.php' ) ); ?>" class="has-update">
								<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 16 16" fill="currentColor" style="width:14px;height:14px">
									<path d="M8.75 2.75a.75.75 0 0 0-1.5 0v5.69L5.03 6.22a.75.75 0 0 0-1.06 1.06l3.5 3.5a.75.75 0 0 0 1.06 0l3.5-3.5a.75.75 0 0 0-1.06-1.06L8.75 8.44V2.75z"/>
									<path d="M3.5 9.75a.75.75 0 0 0-1.5 0v1.5A2.75 2.75 0 0 0 4.75 14h6.5A2.75 2.75 0 0 0 14 11.25v-1.5a.75.75 0 0 0-1.5 0v1.5c0 .69-.56 1.25-1.25 1.25h-6.5c-.69 0-1.25-.56-1.25-1.25v-1.5z"/>
								</svg>
								Update Now
							</a>
						<?php else : ?>
							<a href="<?php echo esc_url( admin_url( 'tools.php?page=oopvulns-vulnerability-scanner' ) ); ?>">
								View Details
							</a>
						<?php endif; ?>
					</div>
				</div>
			</td>
		</tr>
		<?php
	}

	/**
	 * AJAX handler to start a scan and get items to scan.
	 */
	public function ajax_start_scan() {
		check_ajax_referer( 'oopvulns_ajax', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'oopvulns-vulnerability-scanner' ) ) );
		}

		// Get scan type (all, core, plugins, or themes)
		$scan_type = isset( $_POST['scan_type'] )
			? sanitize_text_field( wp_unslash( $_POST['scan_type'] ) )
			: 'all';

		// Get all plugins and themes to scan.
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins = get_plugins();
		$all_themes  = wp_get_themes();

		$items = array();

		// Add WordPress core if requested.
		if ( 'all' === $scan_type || 'core' === $scan_type ) {
			$items[] = array(
				'type' => 'core',
				'name' => 'WordPress Core',
				'slug' => 'wordpress',
			);
		}

		// Add plugins if requested.
		if ( 'all' === $scan_type || 'plugins' === $scan_type ) {
			foreach ( $all_plugins as $plugin_file => $plugin_data ) {
				$slug = $this->scanner->get_plugin_slug( $plugin_file );
				$items[] = array(
					'type' => 'plugin',
					'name' => $plugin_data['Name'],
					'slug' => $slug,
				);
			}
		}

		// Add themes if requested.
		if ( 'all' === $scan_type || 'themes' === $scan_type ) {
			foreach ( $all_themes as $slug => $theme ) {
				$items[] = array(
					'type' => 'theme',
					'name' => $theme->get( 'Name' ),
					'slug' => $slug,
				);
			}
		}

		wp_send_json_success( array(
			'items' => $items,
			'total' => count( $items ),
		) );
	}

	/**
	 * AJAX handler to scan a single item.
	 */
	public function ajax_scan_item() {
		check_ajax_referer( 'oopvulns_ajax', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'oopvulns-vulnerability-scanner' ) ) );
		}

		$type  = isset( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : '';
		$slug  = isset( $_POST['slug'] ) ? sanitize_text_field( wp_unslash( $_POST['slug'] ) ) : '';
		$index = absint( $_POST['index'] ?? 0 );
		$total = absint( $_POST['total'] ?? 0 );

		try {
			switch ( $type ) {
				case 'core':
					$result = $this->scanner->scan_core( false );
					break;

				case 'plugin':
					// Scan individual plugin and update cache.
					$result = $this->scan_single_plugin( $slug );
					break;

				case 'theme':
					// Scan individual theme and update cache.
					$result = $this->scan_single_theme( $slug );
					break;

				default:
					wp_send_json_error( array( 'message' => __( 'Invalid type.', 'oopvulns-vulnerability-scanner' ) ) );
			}

			// If this is the last item, update the summary and last scan time.
			if ( $index >= $total - 1 ) {
				$this->scanner->update_last_scan_time();
			}

			wp_send_json_success( array(
				'result' => $result,
				'index' => $index,
				'total' => $total,
			) );

		} catch ( \Exception $e ) {
			wp_send_json_error( array( 'message' => $e->getMessage() ) );
		}
	}

	/**
	 * Scan a single plugin and update the cached results.
	 *
	 * @param string $slug Plugin slug.
	 * @return array Scan result.
	 */
	private function scan_single_plugin( $slug ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins = get_plugins();
		
		foreach ( $all_plugins as $plugin_file => $plugin_data ) {
			$plugin_slug = $this->scanner->get_plugin_slug( $plugin_file );
			if ( $plugin_slug === $slug ) {
				$version = $plugin_data['Version'];
				$vulns = $this->scanner->api_client->get_plugin_vulnerabilities( $slug, $version, false );

				$result = array(
					'name'            => $plugin_data['Name'],
					'version'         => $version,
					'slug'            => $slug,
					'file'            => $plugin_file,
					'active'          => is_plugin_active( $plugin_file ),
					'vulnerabilities' => $vulns ?: array(),
					'vulnerable'      => ! empty( $vulns ),
					'count'           => $vulns ? count( $vulns ) : 0,
				);

				// Update the cached plugins result.
				$plugins_result = get_option( 'oopvulns_plugins_result', array( 'plugins' => array() ) );
				$plugins_result['plugins'][ $slug ] = $result;
				
				// Recalculate summary.
				$total_vulns = 0;
				$vulnerable_count = 0;
				foreach ( $plugins_result['plugins'] as $p ) {
					if ( ! empty( $p['vulnerable'] ) ) {
						$vulnerable_count++;
						$total_vulns += $p['count'];
					}
				}
				$plugins_result['total'] = count( $plugins_result['plugins'] );
				$plugins_result['vulnerable'] = $vulnerable_count;
				$plugins_result['total_vulnerabilities'] = $total_vulns;

				update_option( 'oopvulns_plugins_result', $plugins_result );

				return $result;
			}
		}

		return array( 'error' => 'Plugin not found' );
	}

	/**
	 * Scan a single theme and update the cached results.
	 *
	 * @param string $slug Theme slug.
	 * @return array Scan result.
	 */
	private function scan_single_theme( $slug ) {
		$all_themes = wp_get_themes();
		$current_theme = wp_get_theme();

		if ( isset( $all_themes[ $slug ] ) ) {
			$theme = $all_themes[ $slug ];
			$version = $theme->get( 'Version' );
			$vulns = $this->scanner->api_client->get_theme_vulnerabilities( $slug, $version, false );

			$result = array(
				'name'            => $theme->get( 'Name' ),
				'version'         => $version,
				'slug'            => $slug,
				'active'          => $slug === $current_theme->get_stylesheet(),
				'vulnerabilities' => $vulns ?: array(),
				'vulnerable'      => ! empty( $vulns ),
				'count'           => $vulns ? count( $vulns ) : 0,
			);

			// Update the cached themes result.
			$themes_result = get_option( 'oopvulns_themes_result', array( 'themes' => array() ) );
			$themes_result['themes'][ $slug ] = $result;

			// Recalculate summary.
			$total_vulns = 0;
			$vulnerable_count = 0;
			foreach ( $themes_result['themes'] as $t ) {
				if ( ! empty( $t['vulnerable'] ) ) {
					$vulnerable_count++;
					$total_vulns += $t['count'];
				}
			}
			$themes_result['total'] = count( $themes_result['themes'] );
			$themes_result['vulnerable'] = $vulnerable_count;
			$themes_result['total_vulnerabilities'] = $total_vulns;

			update_option( 'oopvulns_themes_result', $themes_result );

			return $result;
		}

		return array( 'error' => 'Theme not found' );
	}
}
