/**
 *  OOPVulns  - Vulnerability Scanner - Admin JavaScript
 * Handles interactive functionality with accessibility support
 */

(function($) {
	'use strict';

	const OOPVulns = {
		/**
		 * Initialize the plugin
		 */
		init: function() {
			this.handleScheduleChange();
			this.handleFormValidation();
			this.handleAccessibility();
			this.initAnnouncer();
			this.initCollapsibleCards();
			this.initScanHandler();
		},

		/**
		 * Initialize collapsible vulnerability cards
		 */
		initCollapsibleCards: function() {
			const self = this;
			
			// Handle collapsible card headers
			$(document).on('click', '.oopvulns-collapsible .oopvulns-card-header', function(e) {
				e.preventDefault();
				const $header = $(this);
				const $card = $header.closest('.oopvulns-card');
				const $body = $card.children('.oopvulns-card-body');
				const isExpanded = $header.attr('aria-expanded') === 'true';
				
				// Toggle state for this card only
				$header.attr('aria-expanded', !isExpanded);
				$card.toggleClass('is-expanded', !isExpanded);
				
				if (isExpanded) {
					$body.attr('hidden', true);
				} else {
					$body.removeAttr('hidden');
				}
				
				// Announce state change
				const state = !isExpanded ? 'expanded' : 'collapsed';
				const cardTitle = $card.find('h3').first().text().trim();
				self.announce(cardTitle + ' ' + state);
			});
			
			// Handle keyboard navigation
			$(document).on('keydown', '.oopvulns-collapsible .oopvulns-card-header', function(e) {
				if (e.key === 'Enter' || e.key === ' ') {
					e.preventDefault();
					$(this).trigger('click');
				}
			});
		},

		/**
		 * Initialize scan handler with progress bar
		 */
		initScanHandler: function() {
			const self = this;
			const $form = $('#oopvulns-scan-form');
			const $btn = $('#oopvulns-scan-btn');
			const $progress = $('#oopvulns-scan-progress');
			const $progressBar = $('#oopvulns-progress-bar');
			const $progressText = $('#oopvulns-progress-text');
			const $progressCount = $('#oopvulns-progress-count');
			const $progressItem = $('#oopvulns-progress-item');

			if (!$form.length) return;

			$form.on('submit', function(e) {
				e.preventDefault();
				self.startScan();
			});
		},

		/**
		 * Start the vulnerability scan with progress tracking
		 */
		startScan: function() {
			const self = this;
			const $btn = $('#oopvulns-scan-btn');
			const $progress = $('#oopvulns-scan-progress');
			const $progressBar = $('#oopvulns-progress-bar');
			const $progressText = $('#oopvulns-progress-text');
			const $progressCount = $('#oopvulns-progress-count');
			const $progressItem = $('#oopvulns-progress-item');
			const scanType = $('#scan-type').val() || 'all';

			// Disable button and show progress
			$btn.prop('disabled', true).addClass('scanning');
			$progress.slideDown(300);

			// Start scan - get items to scan
			$.ajax({
				url: oopvulnsData.ajaxUrl,
				type: 'POST',
				data: {
					action: 'oopvulns_start_scan',
					nonce: oopvulnsData.nonce,
					scan_type: scanType
				},
				success: function(response) {
					if (response.success && response.data.items) {
						const items = response.data.items;
						const total = items.length;
						
						$progressText.text(oopvulnsData.i18n.scanning || 'Scanning...');
						$progressCount.text('0 / ' + total);
						
						// Scan items one by one
						self.scanItems(items, 0, total);
					} else {
						self.scanError(oopvulnsData.i18n.scanError || 'Scan failed.');
					}
				},
				error: function() {
					self.scanError(oopvulnsData.i18n.scanError || 'Scan failed.');
				}
			});
		},

		/**
		 * Recursively scan items
		 */
		scanItems: function(items, index, total) {
			const self = this;
			const $progressBar = $('#oopvulns-progress-bar');
			const $progressCount = $('#oopvulns-progress-count');
			const $progressItem = $('#oopvulns-progress-item');

			if (index >= total) {
				// Scan complete
				self.scanComplete();
				return;
			}

			const item = items[index];
			const progress = Math.round((index / total) * 100);

			// Update UI
			$progressBar.css('width', progress + '%');
			$progressCount.text((index + 1) + ' / ' + total);
			$progressItem.html('<span class="dashicons dashicons-update rotating"></span> ' + item.name);

			// Scan this item
			$.ajax({
				url: oopvulnsData.ajaxUrl,
				type: 'POST',
				data: {
					action: 'oopvulns_scan_item',
					nonce: oopvulnsData.nonce,
					type: item.type,
					slug: item.slug,
					index: index,
					total: total
				},
				success: function(response) {
					if (response.success) {
						// Move to next item
						setTimeout(function() {
							self.scanItems(items, index + 1, total);
					}, 500); // Delay for visual feedback
				} else {
					// Continue even if one item fails
					console.warn('Failed to scan ' + item.name);
					setTimeout(function() {
						self.scanItems(items, index + 1, total);
					}, 500);
					}
				},
				error: function() {
					// Continue even if one item fails
					console.warn('Error scanning ' + item.name);
					setTimeout(function() {
						self.scanItems(items, index + 1, total);
					}, 500);
				}
			});
		},

		/**
		 * Handle scan completion
		 */
		scanComplete: function() {
			const $btn = $('#oopvulns-scan-btn');
			const $progressBar = $('#oopvulns-progress-bar');
			const $progressText = $('#oopvulns-progress-text');
			const $progressItem = $('#oopvulns-progress-item');

			$progressBar.css('width', '100%');
			$progressText.text(oopvulnsData.i18n.scanComplete || 'Scan complete!');
			$progressItem.html('<span class="dashicons dashicons-yes-alt"></span> ' + (oopvulnsData.i18n.scanComplete || 'Scan complete!'));

			// Show completion message and reload after delay
			setTimeout(function() {
				location.reload();
			}, 1500);
		},

		/**
		 * Handle scan error
		 */
		scanError: function(message) {
			const $btn = $('#oopvulns-scan-btn');
			const $progress = $('#oopvulns-scan-progress');
			const $progressText = $('#oopvulns-progress-text');
			const $progressItem = $('#oopvulns-progress-item');

			$progressText.text(message);
			$progressItem.html('<span class="dashicons dashicons-warning"></span> ' + message);
			$btn.prop('disabled', false).removeClass('scanning');

			setTimeout(function() {
				$progress.slideUp(300);
			}, 3000);
		},

		/**
		 * Handle notification schedule changes
		 */
		handleScheduleChange: function() {
			const $schedule = $('#notification_schedule');
			const $dayRow = $('#notification-day-row');

			if ($schedule.length) {
				// Initial state
				this.toggleDayRow($schedule.val());

				// On change
				$schedule.on('change', function() {
					OOPVulns.toggleDayRow($(this).val());
				});
			}
		},

		/**
		 * Toggle day of week row visibility
		 */
		toggleDayRow: function(schedule) {
			const $dayRow = $('#notification-day-row');
			if (schedule === 'daily') {
				$dayRow.slideUp(200).attr('aria-hidden', 'true');
				$('#notification_day').prop('disabled', true);
			} else {
				$dayRow.slideDown(200).attr('aria-hidden', 'false');
				$('#notification_day').prop('disabled', false);
			}
		},

		/**
		 * Handle form validation
		 */
		handleFormValidation: function() {
			const $form = $('.oopvulns-form');

			if ($form.length) {
				$form.on('submit', function(e) {
					const $email = $('#notification_email');
					
					// Validate email
					if ($email.length && $email.val()) {
						const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
						if (!emailPattern.test($email.val())) {
							e.preventDefault();
							OOPVulns.showError(
								$email,
								oopvulnsData.i18n.invalidEmail || 'Please enter a valid email address.'
							);
							$email.focus();
							return false;
						}
					}
				});

				// Clear error on input
				$('.regular-text, select').on('input change', function() {
					OOPVulns.clearError($(this));
				});
			}
		},

		/**
		 * Show validation error
		 */
		showError: function($field, message) {
			this.clearError($field);

			$field.attr('aria-invalid', 'true');
			$field.addClass('oopvulns-error');

			const $error = $('<p>')
				.attr({
					'class': 'oopvulns-error-message',
					'role': 'alert',
					'id': $field.attr('id') + '-error'
				})
				.text(message);

			$field.attr('aria-describedby', $error.attr('id'));
			$field.after($error);

			// Announce to screen readers
			this.announce(message);
		},

		/**
		 * Clear validation error
		 */
		clearError: function($field) {
			$field.attr('aria-invalid', 'false');
			$field.removeClass('oopvulns-error');
			const errorId = $field.attr('id') + '-error';
			$('#' + errorId).remove();
			$field.removeAttr('aria-describedby');
		},

		/**
		 * Handle keyboard accessibility
		 */
		handleAccessibility: function() {
			// Add keyboard navigation for cards
			$('.oopvulns-card').each(function() {
				const $card = $(this);
				const $links = $card.find('a');

				// Make cards keyboard accessible if they contain links
				if ($links.length === 1) {
					$card.attr({
						'tabindex': '0',
						'role': 'button'
					});

					$card.on('keydown', function(e) {
						if (e.key === 'Enter' || e.key === ' ') {
							e.preventDefault();
							$links[0].click();
						}
					});
				}
			});

			// Handle escape key for dismissible elements
			$(document).on('keydown', function(e) {
				if (e.key === 'Escape') {
					$('.notice.is-dismissible').find('.notice-dismiss').trigger('click');
				}
			});

			// Improve focus visibility
			$('a, button, input, select, textarea').on('focus', function() {
				$(this).addClass('oopvulns-has-focus');
			}).on('blur', function() {
				$(this).removeClass('oopvulns-has-focus');
			});
		},

		/**
		 * Initialize live region announcer for screen readers
		 */
		initAnnouncer: function() {
			if ($('#oopvulns-announcer').length === 0) {
				$('body').append(
					$('<div>')
						.attr({
							'id': 'oopvulns-announcer',
							'aria-live': 'polite',
							'aria-atomic': 'true',
							'class': 'screen-reader-text'
						})
				);
			}
		},

		/**
		 * Announce message to screen readers
		 */
		announce: function(message) {
			const $announcer = $('#oopvulns-announcer');
			if ($announcer.length) {
				$announcer.text(message);
				
				// Clear after 1 second
				setTimeout(function() {
					$announcer.empty();
				}, 1000);
			}
		},

		/**
		 * Handle loading states
		 */
		setLoading: function($button, isLoading) {
			if (isLoading) {
				$button.prop('disabled', true);
				$button.attr('aria-busy', 'true');
				$button.data('original-text', $button.text());
				
				const loadingText = oopvulnsData.i18n.scanning || 'Loading...';
				$button.html(
					'<span class="dashicons dashicons-update oopvulns-spinning"></span> ' + loadingText
				);
				
				this.announce(loadingText);
			} else {
				$button.prop('disabled', false);
				$button.attr('aria-busy', 'false');
				$button.html($button.data('original-text'));
			}
		},

		/**
		 * Show success message
		 */
		showSuccess: function(message) {
			this.showNotice(message, 'success');
		},

		/**
		 * Show error message
		 */
		showErrorMessage: function(message) {
			this.showNotice(message, 'error');
		},

		/**
		 * Show admin notice
		 */
		showNotice: function(message, type) {
			const $notice = $('<div>')
				.attr({
					'class': 'notice notice-' + type + ' is-dismissible',
					'role': 'alert'
				})
				.html('<p>' + message + '</p>');

			$('.wrap h1').first().after($notice);

			// Announce to screen readers
			this.announce(message);

			// Auto-dismiss after 5 seconds
			setTimeout(function() {
				$notice.fadeOut(300, function() {
					$(this).remove();
				});
			}, 5000);

			// Handle dismiss button
			$notice.find('.notice-dismiss').on('click', function() {
				$notice.fadeOut(300, function() {
					$(this).remove();
				});
			});
		}
	};

	// Initialize when document is ready
	$(document).ready(function() {
		OOPVulns.init();
	});

	// Expose to global scope for external use
	window.OOPVulns = OOPVulns;

})(jQuery);

// CSS for spinning animation (injected dynamically if needed)
if (typeof jQuery !== 'undefined') {
	jQuery(document).ready(function($) {
		const css = `
			@keyframes oopvulns-spin {
				from { transform: rotate(0deg); }
				to { transform: rotate(360deg); }
			}
			.oopvulns-spinning {
				animation: oopvulns-spin 1s linear infinite;
				display: inline-block;
			}
			.oopvulns-error {
				border-color: #dc3545 !important;
				box-shadow: 0 0 0 1px #dc3545 !important;
			}
			.oopvulns-error-message {
				color: #dc3545;
				font-size: 13px;
				margin: 5px 0 0;
			}
			.oopvulns-has-focus {
				outline: 2px solid #FCDD56;
				outline-offset: 2px;
			}
		`;

		if ($('#oopvulns-dynamic-css').length === 0) {
			$('head').append('<style id="oopvulns-dynamic-css">' + css + '</style>');
		}
	});
}
